/*
 * This is a wrapper program to let LyX/Cygwin use either a native Win32 or
 * Cygwin version of epstopdf (coming with either teTeX or MiKTeX).
 *
 * This program simply changes all input pathnames into Windows or Cygwin
 * pathnames depending on which version of epstopdf is being used.
 *
 * Author: Enrico Forestieri
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>
#include <sys/cygwin.h>
#include <sys/wait.h>

#define strfcpy(A,B,C) strncpy(A,B,C), *(A+(C)-1)=0

extern char *__progname;

int main(int ac, char **av)
{
    char *path = getenv("PATH");
    char *dir;
    char epstopdf[3 * PATH_MAX];
    char convbuf[PATH_MAX];
    int fail = 1;
    int size;
    int len;
    int i;
    cygwin_conv_path_t what;

    if (!path || !(path = strdup(path))) {
	fprintf(stderr, "%s: No PATH or out of memory", __progname);
	exit(1);
    }

    for (dir = strtok(path, ":"); dir; dir = strtok(NULL, ":")) {
	strfcpy(convbuf, dir, sizeof(convbuf));
	if (dir[0] != '.' && strncmp(dir, "/usr/local/bin", 14) != 0
		&& strlen(convbuf) + 10 < sizeof(convbuf)) {
	    strcat(convbuf, "/epstopdf");
	    if (access(convbuf, X_OK) == 0) {
		fail = 0;
		break;
	    }
	}
    }

    free(path);
    if (fail) {
	fprintf(stderr, "%s: Cannot find a real epstopdf binary\n", __progname);
	exit(2);
    }

    /* Assume native Win32 binaries are outside the cygwin tree */
    if (strncmp(convbuf, "/usr", 4) == 0 || strncmp(convbuf, "/bin", 4) == 0)
	what = CCP_WIN_A_TO_POSIX | CCP_RELATIVE;
    else
	what = CCP_POSIX_TO_WIN_A | CCP_RELATIVE;

    snprintf(epstopdf, sizeof(epstopdf), "\"%s\"", convbuf);

    for (i = 1; i < ac; ++i) {
	char *opt = strdup(av[i]);
	char *val = 0;
	if (!opt) {
	    fail = 1;
	    break;
	}
	if (opt[0] == '-') {
	    opt = strtok(opt, "=");
	    val = strtok(NULL, "*");
	    strfcpy(convbuf, opt, sizeof(convbuf));
	} else {
	    convbuf[0] = '"';
	    if (cygwin_conv_path(what, opt, convbuf + 1, sizeof(convbuf) - 2)) {
		fail = 1;
		break;
	    }
	    strncat(convbuf, "\"", sizeof(convbuf) - strlen(convbuf));
	}
	len = strlen(epstopdf);
	size = sizeof(epstopdf) - len;
	len += snprintf(epstopdf + len, size, " %s", convbuf);
	if (len >= sizeof(epstopdf)) {
	    fail = 1;
	    break;
	}
	if (val) {
	    if (cygwin_conv_path(what, val, convbuf, sizeof(convbuf))) {
		fail = 1;
		break;
	    }
	    size = sizeof(epstopdf) - len;
	    len += snprintf(epstopdf + len, size, "=\"%s\"", convbuf);
	    if (len >= sizeof(epstopdf)) {
		fail = 1;
		break;
	    }
	}
	free(opt);
    }

    if (fail) {
	fprintf(stderr, "%s: Command too long\n", __progname);
	exit(3);
    }

    fail = system(epstopdf);

    return WEXITSTATUS(fail);
}
