/*
 *  Author: Enrico Forestieri <forenr@lyx.org>
 *  Created:  Sat Sep  4 16:47:24 2004
 *  Modified: Fri Jun 10 09:51:41 2005
 *     - If no flags given, -q implied.
 *  Modified: Sun Apr 16 20:30:36 2006
 *     - Account for spaces in gsprint path.
 *  Modified: Sun Jun 15 21:10:24 2014
 *     - Use the new cygwin_conv_path API.
 *     - Add --help option.
 *
 *  Modify and redistribute freely. Use at your own risk.
 *  This is distributed as part of the LyX/Cygwin package.
 *
 *  Program to print postscript files to a windows printer through the cygwin
 *  lpr or gsprint ( http://www.cs.wisc.edu/~ghost/gsview/gsprint.htm ).
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <getopt.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/cygwin.h>

#define LINSIZE 260
#define BUFSIZE 1024
#define MAXNVER 10

extern char *__progname;

struct option longopts[] = {
    {"help", no_argument, NULL, 0},
    {0, no_argument, NULL, 0}
};


void convert_path(cygwin_conv_path_t what, char const *from, char *to,
		  size_t to_size)
{
    ssize_t size;
    size = cygwin_conv_path(what, from, NULL, 0);
    if (size < 0 || size > to_size)
    {
	to[0] = 0;
	if (size > to_size)
	    errno = ENAMETOOLONG;
	perror("cygwin_conv_path");
	return;
    }
    if (cygwin_conv_path(what, from, to, to_size))
    {
	to[0] = 0;
	perror("cygwin_conv_path");
    }
}


void conv_to_posix_path(char const *from, char *to, size_t to_size)
{
    convert_path(CCP_WIN_A_TO_POSIX | CCP_RELATIVE, from, to, to_size);
}


void conv_to_win_path(char const * from, char *to, size_t to_size)
{
    convert_path(CCP_POSIX_TO_WIN_A | CCP_RELATIVE, from, to, to_size);
}


int main(int ac, char **av)
{
    int c;
    int Dflg=0, Pflg=0, hflg=0, lflg=0, qflg=0;
    int errflg=0;
    int retval=0;
    FILE *fp;
    char *device=NULL;
    char *file=NULL;
    char pbuf[LINSIZE];
    char cbuf[BUFSIZE];
    char gsprint[LINSIZE];

    while ((c = getopt_long(ac, av, "DP:d:hlq", longopts, NULL)) != EOF)
    {
	switch (c)
	{
	    case 'D':
		Dflg++;
		break;
	    case 'P':
	    case 'd':
		Pflg++; 
		device = optarg;
		break;
	    case 'h':
		hflg++;
		break;
	    case 'l':
		lflg++;
		break;
	    case 'q':
		qflg++;
		break;
	    default:
		errflg++;
	}
    }

    if (errflg || (qflg && (Dflg || Pflg || hflg || lflg)))
    {
	fprintf(stderr,"Prints the given files to a windows printer by using the cygwin lpr or gsprint.\n\n");
	fprintf(stderr,"Usage: lp [-D] [-d device ] [-h] [-l] [-P device ] <files>...\n");
	fprintf(stderr,"   or: lp -q <postscript files>...\n\n");
	fprintf(stderr,"With the first form, merely calls lpr for each given file (see manpage for lpr).\n");
	fprintf(stderr,"With the second form (or if no switches are given), calls gsprint showing the\nprinter setup dialog. If gsprint is not found, defaults to lpr with no args.\n");
	exit(1);
    }

    if (!Dflg && !Pflg && !hflg && !lflg)
	qflg++;

    if (qflg)
    {	/* Look for gsview in the registry */
	gsprint[0] = 0;
	if ((fp=popen("regtool -q list /HKLM/SOFTWARE/Ghostgum/Gsview","r")))
	{   /* Scan all versions, keep the last good one */
	    int v=0;
	    char *ver[MAXNVER];
	    while (fgets(pbuf,sizeof(pbuf),fp) && v<MAXNVER)
	    {
		pbuf[strlen(pbuf)-1] = 0;
		if (pbuf[0])
		{
		    ver[v] = malloc(strlen(pbuf)+1);
		    strcpy(ver[v],pbuf);
		    v++;
		}
	    }
	    pclose(fp);
	    if (v)
	    {
		int i,l;
		strcpy(cbuf,"regtool -q get \"/HKLM/SOFTWARE/Ghostgum/Gsview/");
		l=strlen(cbuf);
		for (i=0; i<v; i++)
		{
		    cbuf[l] = 0;
		    strcat(cbuf,ver[i]);
		    strcat(cbuf,"\"");
		    if ((fp=popen(cbuf,"r")))
		    {
			if (fgets(pbuf,sizeof(pbuf),fp))
			{
			    char tbuf[LINSIZE];
			    pbuf[strlen(pbuf)-1] = 0;
			    if (pbuf[0])
			    {
				struct stat st;
				conv_to_posix_path(pbuf,tbuf,LINSIZE-20);
				strcat(tbuf,"/gsview/gsprint.exe");
				if (stat(tbuf,&st)==0)
				    strcpy(gsprint,tbuf);
			    }
			}
			pclose(fp);
		    }
		    free(ver[i]);
		}
	    }
	}
	if (gsprint[0])
	{
	    strcpy(cbuf,"\"");
	    strcat(cbuf,gsprint);
	    strcat(cbuf,"\" -q -query ");
	}
	else
	{   /* No gsview, defaults to lpr */
	    strcpy(cbuf,"/usr/bin/lpr ");
	}
    }
    else
    {
	strcpy(cbuf,"/usr/bin/lpr");
	if (Dflg)
	    strcat(cbuf," -D");
	if (Pflg)
	{
	    strcat(cbuf," -P ");
	    strcat(cbuf,device);
	}
	if (hflg)
	    strcat(cbuf," -h");
	if (lflg)
	    strcat(cbuf," -l");
	strcat(cbuf," ");
    }
    
    if (optind==ac)
    {	/* Read from stdin and create a temp file */
	file="/tmp/gsprint.ps";
	if ((fp=fopen(file,"w")))
	{
	    while (fgets(pbuf,sizeof(pbuf),stdin))
		fputs(pbuf,fp);
	    fclose(fp);
	}
	else
	{
	    fprintf(stderr, "%s: cannot open temporary file\n", __progname);
	    exit(1);
	}
	conv_to_win_path(file,pbuf,LINSIZE);
	if (pbuf[0])
	{
	    strcat(cbuf,"'");
	    strcat(cbuf,pbuf);
	    strcat(cbuf,"'");
	    system(cbuf);
	}
	else
	{
	    fprintf(stderr, "%s: unable to print\n", __progname);
	    retval=1;
	}
	remove(file);
    }
    else
    {	/* Remaining args on cmd line are treated as files to print */
	int l = strlen(cbuf);
	while (optind<ac)
	{
	    cbuf[l] = 0;
	    conv_to_win_path(av[optind],pbuf,LINSIZE);
	    if (pbuf[0])
	    {
		strcat(cbuf,"'");
		strcat(cbuf,pbuf);
		strcat(cbuf,"'");
		system(cbuf);
	    }
	    else
	    {
		fprintf(stderr, "%s: cannot print file \"%s\"\n",
			__progname, av[optind]);
		retval=1;
	    }
	    optind++;
	}
    }

    return retval;
}
