/* -*-C-*- hash.c */
/*-->hash*/
/**********************************************************************/
/******************************* hash *********************************/
/**********************************************************************/

/***********************************************************************

This file provides a GENERIC hash table interface which permits the
simultaneous maintenance of multiple hash tables.

The UNIX hcreate(), hdestroy(), and hsearch() utilities support only one
table at a time, and in any event, they are NOT part of Standard C.

Declarations:
	#include "hash.h"
	...
	SOMETHING	*data;
	HASH_ENTRY	*h;
	HASH_TABLE	*table;

To create a table:
	table = hash_alloc(minimum_number_of_elements, case_sensitive);

To install an entry with key "key string" and data:
	h = hash_install("key string", (void*)data, table);
	if (h == (HASH_ENTRY*)NULL)
	    error("Hash table is full");

To lookup an entry with key "key string":
	h = hash_lookup("key string", table);
	if (h == (HASH_ENTRY*)NULL)
	    error("Key string not found in hash table");
	else if (h->hash_key == (const char*)NULL)
	    maybe_create_new_entry(h);
	else
	    data = (SOMETHING*)h->hash_data;

To loop over elements of a table (in no particular order):
	for (h = hash_next(1,table); h != (HASH_ENTRY*)NULL;
		h = hash_next(0,table))
	...

To destroy a hash table:
	hash_free(table);

A full hash table cannot arise unless the initial size passed to
hash_alloc() is less than the number of elements installed.

Statistics on the hashing are maintained in HASH_TABLE members
n_compare, n_install, n_lookup, and n_probe; these record the number of
calls to the string comparison routine, hash_compare(), to
hash_install(), to hash_lookup(), and the number of entries probed in
the overflow chains.  Linear probing is used, and the table loading is
limited to about 80% full.  If the hash function is behaving properly,
the ratio n_compare/n_lookup should be less than 2.

The maximum number of entries in the hash table is limited by the size
of the typedef HASH_INT in hash.h; it must be an unsigned integer type.

Table memory is dynamically managed using xmalloc() and xfree();
execution is terminated if there is insufficient memory.

A test program that exercises these functions is stored in readps.c in
the Utah DVI Driver Family Version 3.x (or later).

***********************************************************************/

#include <config.h>
#include "xstring.h"
#include "xalloc.h"
#include "xctype.h"
#include "hash.h"


#if 0
/* For debugging: */
#define DBGOPT(flag)	(debug_code & (flag))
extern UNSIGN32		debug_code;	/* 0 for no debug output */
extern FILE *stddbg;
#endif


RCSID("$Id: hash.c,v 1.8 1993/12/29 19:24:11 beebe Exp beebe $")


static HASH_INT	hash ARGS((const char *key_, const HASH_TABLE *table_));
static long	next_prime ARGS((long n_));

/*-->hash*/
/**********************************************************************/
/******************************* hash *********************************/
/**********************************************************************/

static HASH_INT
#if defined(HAVE_STDC)
hash(
const char		*key,
const HASH_TABLE	*table	/* hash_table created by hash_alloc() */
)
#else
hash(key,table)
const char		*key;
const HASH_TABLE	*table;	/* hash_table created by hash_alloc() */
#endif
{
    HASH_INT		hash_code;

    if (table == (HASH_TABLE*)NULL)
	return (0);			/* NULL table: no index available */

    /* This hash function gives a reasonably even distribution when
       tested against the UNIX spelling dictionary, /usr/dict/words. */
    if (table->hash_compare == strcmp)	/* case sensitive */
    {
	for (hash_code = 0; *key; ++key)
	    hash_code = (hash_code << 1) ^ (0xff & *key);
    }
    else				/* case insensitive */
    {
	for (hash_code = 0; *key; ++key)
	    hash_code = (hash_code << 1) ^ (0xff &
		(HASH_INT)(islower(*key) ? (int)*key : tolower(*key)));
    }
    return (hash_code % (table->hash_size));
}

/*-->hash_alloc*/
/**********************************************************************/
/**************************** hash_alloc ******************************/
/**********************************************************************/

HASH_TABLE*
#if defined(HAVE_STDC)
hash_alloc(
HASH_INT	size,	/* minimum required size of hash table */
int		case_sensitive	/* 0 if not case sensitive, else non-0 */
)
#else
hash_alloc(size,case_sensitive)/* allocate, initialize, and return hash table */
HASH_INT	size;	/* minimum required size of hash table */
int		case_sensitive;	/* 0 if not case sensitive, else non-0 */
#endif
{
    HASH_INT	k;
    HASH_TABLE	*table;

    size = (HASH_INT)next_prime((long)size);	/* make hash size a prime for */
						/* good hash performance */

    /* allocate hash table */
    table = (HASH_TABLE*)xmalloc(size*sizeof(HASH_TABLE));

    /* initialize hash table */
    table->hash_index = (HASH_INT)0;
    table->hash_size = (HASH_INT)size;
    table->hash_table = (HASH_ENTRY*)xmalloc(size*sizeof(HASH_ENTRY));
    table->hash_compare = case_sensitive ? strcmp : stricmp;
    table->n_compare = 0L;	/* clear statistics */
    table->n_install = 0L;
    table->n_lookup = 0L;
    table->n_probe = 0L;
    table->max_chain = 0L;

    /* initialize hash entries */
    for (k = 0; k < size; ++k)
    {
	table->hash_table[k].hash_key = (const char*)NULL;
	table->hash_table[k].hash_data = (VOIDP)NULL;
    }

    return (table);
}

/*-->hash_free*/
/**********************************************************************/
/***************************** hash_free ******************************/
/**********************************************************************/

void
#if defined(HAVE_STDC)
hash_free(
HASH_TABLE	*table
)
#else
hash_free(table)
HASH_TABLE	*table;
#endif
{
    if (table != (HASH_TABLE*)NULL)
    {
	xfree((VOIDP)table->hash_table);
	xfree((VOIDP)table);
    }
}

/*-->hash_install*/
/**********************************************************************/
/*************************** hash_install *****************************/
/**********************************************************************/

HASH_ENTRY*
#if defined(HAVE_STDC)
hash_install(
const char	*key,		/* key string tagging data */
VOIDP		data,		/* data to install */
HASH_TABLE	*table		/* hash_table created by hash_alloc() */
)
#else
hash_install(key,data,table)
const char	*key;		/* key string tagging data */
VOIDP		data;		/* data to install */
HASH_TABLE	*table;		/* hash_table created by hash_alloc() */
#endif
{
    HASH_ENTRY	*h;

    h = hash_lookup(key,table);
    if (h != (HASH_ENTRY*)NULL)
    {
	table->n_install++;		/* update statistics */
	h->hash_key = key;
	h->hash_data = data;
    }
    return (h);
}

/*-->hash_lookup*/
/**********************************************************************/
/**************************** hash_lookup *****************************/
/**********************************************************************/

HASH_ENTRY*
#if defined(HAVE_STDC)
hash_lookup(
				/* or NULL if table full */
const char	*key,		/* key string to search for */
HASH_TABLE	*table		/* hash_table created by hash_alloc() */
)
#else
hash_lookup(key,table)		/* return (maybe empty) entry for key, */
				/* or NULL if table full */
const char	*key;		/* key string to search for */
HASH_TABLE	*table;		/* hash_table created by hash_alloc() */
#endif
{
    HASH_ENTRY	*h = (HASH_ENTRY*)NULL;
    HASH_INT	n;
    HASH_INT	probes;

    if (table == (HASH_TABLE*)NULL)	/* NULL table has no entries */
	return ((HASH_ENTRY*)NULL);

    table->n_lookup++;			/* update statistics */

    n = hash(key,table);		/* initial hash index */

    for (probes = 1; probes <= table->hash_size; ++probes)
    {
	table->n_probe++;		/* update statistics */

	if (table->hash_table[n].hash_key == (const char*)NULL)
	{
	    h = &table->hash_table[n];	/* not in table */
	    break;
	}
	else if ((table->hash_compare)(table->hash_table[n].hash_key, key) == 0)
	{
	    table->n_compare++;	/* update statistics */
	    h = &table->hash_table[n];	/* have entry in table */
	    break;
	}
	else
	{
	    table->n_compare++;	/* update statistics */
	    n++;			/* linear probe to next entry */
	    if (n >= table->hash_size)
		n = (HASH_INT)0;
	}
    }
    if (probes > table->max_chain)
	table->max_chain = probes;
#if 0
    if (DBGOPT(0x20000))
	fprintf(stddbg,
		"hash_lookup(\"%s\",table): probes = %ld  n_probe = %ld  \
n_compare = %ld  hash = %ld  key = \"%s\"\n",
		key, (long)probes, (long)table->n_probe,
		(long)table->n_compare, (long)n,
		(table->hash_table[n].hash_key == (const char*)NULL) ?
		"<EMPTY>" : table->hash_table[n].hash_key);
#endif
    return (h);
}

/*-->hash_next*/
/**********************************************************************/
/***************************** hash_next ******************************/
/**********************************************************************/

HASH_ENTRY*
#if defined(HAVE_STDC)
hash_next(
int		first,
HASH_TABLE	*table
)
#else
hash_next(first,table)		/* return next entry from table, or NULL */
int		first;
HASH_TABLE	*table;
#endif
{
    if (table == (HASH_TABLE*)NULL)
	return ((HASH_ENTRY*)NULL);	/* NULL table is always empty */
    if (first)
	table->hash_index = 0;
    while (table->hash_index < table->hash_size)
    {
	if (table->hash_table[table->hash_index].hash_key !=
	    (const char*)NULL)
	    return (&table->hash_table[table->hash_index++]);
	else
	    table->hash_index++;
    }
    return ((HASH_ENTRY*)NULL);		/* end of table reached */
}

/*-->next_prime*/
/**********************************************************************/
/**************************** next_prime ******************************/
/**********************************************************************/

#ifdef FALSE
#undef FALSE
#endif /* FALSE */
#define FALSE 0

static long
#if defined(HAVE_STDC)
next_prime(
long n
)
#else
next_prime(n)			/* return next prime at or after n */
long n;
#endif
{
    long prime;				/* tentative prime */
    long factor;			/* prime factor */
    int is_prime;			/* 'prime' is a prime number */

    n = (n < 0L) ? -n : n;		/* be safe -- force n positive */
    prime = 2L*(n/2L) + 1L;		/* next odd number */
    is_prime = (prime <= 3L);
    while (!is_prime)
    {
	factor = 5L;
	is_prime = (prime % 2L) && (prime % 3L);
	while (is_prime && (factor*factor <= prime))
	{
	    if ((prime % factor) == 0L)
		is_prime = FALSE;
	    else if ((prime % (factor + 2L)) == 0L)
		is_prime = FALSE;
	    else	/* factor+4 is divisible by 3 (every 3rd odd is) */
		factor += 6L;
	}
	if (!is_prime)
		prime += 2L;
    }
    return (prime);
}
