### ====================================================================
###  @Awk-file{
###     author          = "Nelson H. F. Beebe",
###     version         = "0.00",
###     date            = "05 November 1994",
###     time            = "10:34:48 MST",
###     filename        = "biblabel.awk",
###     address         = "Center for Scientific Computing
###                        Department of Mathematics
###                        University of Utah
###                        Salt Lake City, UT 84112
###                        USA",
###     telephone       = "+1 801 581 5254",
###     FAX             = "+1 801 581 4148",
###     checksum        = "08279 878 3230 26754",
###     email           = "beebe@math.utah.edu (Internet)",
###     codetable       = "ISO/ASCII",
###     keywords        = "bibliography, BibTeX, citation label, LaTeX,
###                        TeX",
###     supported       = "yes",
###     docstring       = "This program reads one or more bibliography
###                        files, and prepares a list of unique
###                        standardized citation labels for selected
###                        document types.  The citation label is formed
###                        by these rules, easily applicable by a
###                        human, or by a computer program like this
###                        one:
###
###                        (1) Take the first author's last name,
###                            dropping apostrophes, Jr/Sr/generation
###                            numbers, and eliminating accents
###                            (e.g. J{\"a}nsch -> Jaensch, and
###                            Jind\v{r}ich -> Jindrich), using
###                            multi-letter transliterations if that is
###                            conventional.  Hyphenated names, like
###                            Baeza-Yates, are preserved in full.
###                        (2) Append a colon.
###                        (3) Append the four-digit year of publication
###                        (4) Append another colon.
###                        (5) Pick the first 3 important words in the
###                            title that begin with a letter, excluding
###                            articles and prepositions and TeX math
###                            mode (e.g. ``On ${C}^1$ interpolating
###                            hierarchical spline bases'' reduces to
###                            IHS), and append that.  If there are
###                            fewer than 3 important words, then use a
###                            shorter string.
###                        (6) If the resulting label is already in use,
###                            add a letter a, b, c, ... to make it
###                            unique.
###
###                        This will produce a label like
###                        Smith:1994:ABC.
###
###                        The reason for including a four-digit year is
###                        that the millenium change is near, and we
###                        expect bibliographies to be in use for many
###                        years hence.  Also, some bibliographies may
###                        be historical, with entries dating back
###                        hundreds of years.  Using a four-digit year
###                        will keep sorts of otherwise identical keys
###                        in chronological order, and putting the year
###                        before the key derived from the title will
###                        facilitate sorting by year.
###
###                        Because any change in citation labels must be
###                        accompanied by a change in citations in all
###                        documents that use the bibliography, it is
###                        not sufficient to just produce a new
###                        bibliography file with changed labels.
###
###                        Instead, the output is a file containing old
###                        and new labels, one per line, suitable for
###                        input to the companion citesub program
###                        (either awk or C versions).
###
###                        To avoid confusion between labels with common
###                        prefixes, such as Smith80 and Smith80a,
###                        citesub will check for leading context of a
###                        left brace, quote, comma, whitespace, or
###                        beginning of line and trailing context of a
###                        right brace, comma, quote, percent,
###                        whitespace, or end of line so as to match
###                        these styles:
###
###                             @Book{Smith:1980:ABC,
###                             crossref = <quote>Smith:1980:ABC<quote>,
###                             crossref = {Smith:1980:ABC},
###                             \cite{Smith:1980:ABC}
###                             \cite{Smith:1980:ABC,Jones:1994:DEF}
###                             \cite{%
###                                     Smith:1980:ABC,%
###                                     Jones:1994:DEF%
###                             }
###
###                        The ignore[] list initialized in
###                        make_ignore_list() contains words which are
###                        to be ignored in preparation of the
###                        alphabetic string from the title words.  It
###                        will be augmented at run-time from two files:
###                        biblabel.ign (intended to be a generic
###                        enhancement), and a file assigned to IGNORE
###                        on the command line.
###
###                        Created labels are guaranteed to be unique
###                        within the input files provided on the
###                        command line.  However, in a larger project,
###                        one may wish to exclude labels that are
###                        already in use in other bibliographies.  To
###                        provide for this, the USED variable can be
###                        set on the command line to define the name of
###                        a file of labels that are already in use.  It
###                        will be augmented by labels from a generic
###                        file biblabel.use.  Lines in these file
###                        consist of pairs of filenames and citation
###                        labels.
###
###                        When a label is found in use, and the current
###                        file matches the in-use label filename, the
###                        label is considered to be unused; otherwise,
###                        repeated runs through this program would keep
###                        changing already-assigned labels.
###
###                        Usage:
###                             nawk -f biblabel.awk
###                                     [-v IGNORE=ignore-file]
###                                     [-v USED=used-labels-file]
###                                     mybib.bib >mybib.sub
###
###                        Examine the output substitutions to decide
###                        whether the generated labels are acceptable,
###                        and make any changes by hand.
###
###                        Tentatively apply the label substitutions and
###                        compare the input and output files:
###
###                        citesub -f mybib.sub mybib.bib >mybib.bib.tmp
###                        diff mybib.bib mybib.bib.tmp
###
###                        citesub -f mybib.sub myfile.ltx >myfile.ltx.tmp
###                        diff myfile.ltx myfile.ltx.tmp
###
###                        If the citesub substitutions are acceptable,
###                        then:
###
###                        mv mybib.bib.tmp mybib.tmp
###                        mv myfile.ltx.tmp myfile.ltx
###
###                        The checksum field above contains a CRC-16
###                        checksum as the first value, followed by the
###                        equivalent of the standard UNIX wc (word
###                        count) utility output of lines, words, and
###                        characters.  This is produced by Robert
###                        Solovay's checksum utility.",
###  }
### ====================================================================

BEGIN 					{ begin_bibliography() }

/^@[A-Za-z][A-Za-z0-9_-]*/		{ begin_entry(); next }

/^ *author *= *"/			{ author_string(); next }

/^ *editor *= *"/			{ editor_string(); next }

/^ *journal *= *j-.*, *$/		{ journal_string(); next }

/^ *number *= *"/			{ number_string(); next }

/^ *pages *= *"/			{ pages_string(); next }

/^ *title *= *"/			{ title_string(); next }

/^ *volume *= *"/			{ volume_string(); next }

/^ *year *= *"/				{ year_string(); next }

/^}/ 					{ end_entry(); next }

					{ next }

END					{ end_bibliography() }

#=======================================================================

function author_string()
{
    s = fix_author_editor(value($0))
    print_debug("authors = <" s ">")
    n = split(s,authors," and ")
    print_debug("First author = <" authors[1] ">")
    n = split(authors[1],names," ")
    lastname = names[n]		# last name of first author

    lastname = fix_accents(lastname)
}


function begin_bibliography()
{
    make_ignore_list()

    extend_ignore_list("biblabel.ign")
    if (IGNORE)
	extend_ignore_list(IGNORE)

    extend_used_list("biblabel.use")
    if (USED)
	extend_used_list(USED)
}



function begin_entry()
{
    lastname = ""
    new_tag = ""
    old_tag = entry_tag($0)
    pages[1] = ""
    title_abbrev = ""
    type = entry_type($0)
    year = "19xx"	# default year so we can always make a citation tag
}


function editor_string()
{
    if (lastname)	# don't set lastname if already set from authors
	return
    s = fix_author_editor(value($0))
    print_debug("editors = <" s ">")
    n = split(s,editors," and ")
    print_debug("First editor = <" editors[1] ">")
    n = split(editors[1],names," ")
    lastname = names[n]		# last name of first author
    lastname = fix_accents(lastname)
}


function end_bibliography()
{
    for (old_tag in old_to_new)
    {
	# NB: we print ALL tags here, even if (old_tag == old_to_new[old_tag]),
	# because we later extract the new tags from the *.sub files to
	# prepare the biblabel.use file
	if (old_tag && old_to_new[old_tag])
	    print_substitution_line(old_tag,old_to_new[old_tag])
    }
}


function end_entry()
{
    if (type == "Article") 		new_tag = tag_Article()
    else if (type == "Book")		new_tag = tag_Book()
    else if (type == "Booklet")		new_tag = tag_Booklet()
    else if (type == "DEAthesis")	new_tag = tag_DEAthesis()
    else if (type == "InBook")		new_tag = tag_InBook()
    else if (type == "InCollection")	new_tag = tag_InCollection()
    else if (type == "InProceedings")	new_tag = tag_InProceedings()
    else if (type == "Manual")		new_tag = tag_Manual()
    else if (type == "MastersThesis")	new_tag = tag_MastersThesis()
    else if (type == "Misc")		new_tag = tag_Misc()
    else if (type == "Periodical")	new_tag = tag_Periodical()
    else if (type == "PhdThesis")	new_tag = tag_PhdThesis()
    else if (type == "Proceedings")	new_tag = tag_Proceedings()
    else if (type == "TechReport")	new_tag = tag_TechReport()
    else if (type == "Unpublished")	new_tag = tag_Unpublished()
    else				new_tag = ""

    print_debug("old_tag = <" old_tag "> type = <" type \
	"> new_tag = <" new_tag ">")
    if (is_tag_in_use(old_tag))		# duplicate tag in input bibliography
	warning("ERROR: duplicate tag <" old_tag "> in input file(s)")
    else
    {
	if (new_tag)			# then generated a valid new tag
	{
	    if ((new_tag in new_to_old) || \
		is_tag_in_use(new_tag)) # then duplicate tag
		make_unique_new_tag()
	    new_to_old[new_tag] = old_tag
	    record_tag_use(FILENAME,old_tag,new_tag)
	}
	else			# remember old tags to check for dups
	    record_tag_use(FILENAME,old_tag,"")
    }
    type = ""			# forget current type and tag
    old_tag = ""		# so as to catch unbalanced braces
}


function entry_tag(s)
{
    # Return the citation tag from a `@EntryType{tag,' line
    k1 = index(s,"{") + 1
    k2 = index(s,",")
    return (substr(s,k1,k2-k1))
}


function entry_type(s)
{
    return (substr(s,2,index(s,"{")-2))
}


function extend_ignore_list(filename)
{
    if (file_exists(filename))
    {
	while (getline word <filename)
	{
	    gsub(/[^a-zA-Z]/,"",word)	# discard all but letters
	    word = tolower(word)	# and convert to lower case
	    ignore[word] = 1
	    print "DEBUG: ignoring word <" word ">"
	}
    }
}


function extend_used_list(filename, line,old_file,old_tag)
{
    if (file_exists(filename))
    {
	while (getline line <filename)
	{
	    split(line,parts,"\t")
	    old_file = parts[1]
	    old_tag = parts[2]
	    gsub(/^[ \t]*/,"",old_tag)	# discard leading and
	    gsub(/[ \t]$/,"",old_tag) 	# trailing space
	    record_tag_use(old_file,old_tag,"")
	    print_debug("excluding old_tag = <" old_tag \
		"> from file = [" old_file "]")
	}
    }
}


function file_exists(filename)
{
    # Unfortunately, getline in both nawk and gawk will hang if its
    # file does not exist, so we need to test for file existence by
    # invoking a shell command, sigh...

    return (system("cat " filename " >/dev/null 2>&1") == 0)
}


function fix_accents(lastname, s)
{
    s = lastname

    # Remove accents from last name, because we need a control-sequence
    # free tag.  The order of some of these is important, so do NOT
    # rearrange them!
    # Eventually, an algorithmic reduction should be applied here instead
    # of all of these substitutions.

    # eliminate acute, grave, circumflex, macron, tilde accents
    gsub(/\\[.'`^=~]/,	"",	s)

    gsub(/\\[uvHtcdb]{/,"",	s) # eliminate most other accents

    gsub(/~/,		" ",	s) # De~Raedt -> De Raedt

    gsub(/{\\i}/,	"i",	s) # dotless i -> i
    gsub(/{\\j}/,	"j",	s) # dotless j -> j
    gsub(/{\\l}/,	"l",	s) # Ko{\\l}os -> Kolos

    gsub(/\\"A/,	"Ae",	s) #
    gsub(/\\"O/,	"Oe",	s) #
    gsub(/\\"U/,	"Ue",	s) #
    gsub(/\\"\\i/,	"i",	s) # Vorono{\"\i} -> Voronoi
    gsub(/\\"a/,	"ae",	s) # J{\"a}nsch -> Jaensch
    gsub(/\\"e/,	"e",	s) # Eli{\"e}ns -> Eliens
    gsub(/\\"i/,	"i",	s) # Bsa{\"i}es -> Bsaies
    gsub(/\\"o/,	"oe",	s) # B{\"o}rgers -> Boergers
    gsub(/\\"u/,	"ue",	s) # R{\"u}de -> Ruede
    gsub(/\\"y/,	"y",	s) # Delabbe{\"y} ->Delabbey
    gsub(/\\"{A}/,	"ae",	s) #
    gsub(/\\"{O}/,	"oe",	s) #
    gsub(/\\"{U}/,	"ue",	s) #
    gsub(/\\"{a}/,	"ae",	s) # J{\"{a}}nsch -> Jaensch
    gsub(/\\"{e}/,	"e",	s) # Eli{\"{e}}ns -> Eliens
    gsub(/\\"{i}/,	"i",	s) # Bsa{\"{i}}es -> Bsaies
    gsub(/\\"{o}/,	"oe",	s) # B{\"{o}}rgers -> Boergers
    gsub(/\\"{u}/,	"ue",	s) # H{\"{u}}bner -> Huebner

    gsub(/\\AA[{}]/,	"Aa",	s) # {\AA}rhus -> Aarhus
    gsub(/\\AE[{}]/,	"Ae",	s) # {\AE}ro -> Aero
    gsub(/\\OE[{}]/,	"Oe",	s) # {\OE}rsted -> Oersted
    gsub(/\\aa[{}]/,	"aa",	s) # Ring\aa -> Ringaa
    gsub(/\\ae[{}]/,	"ae",	s) # D{\ae}hlen -> Daehlen
    gsub(/\\oe[{}]/,	"ae",	s) # St{\oe}ren -> Stoeren
    gsub(/\\o[{}]/,	"oe",	s) # Bj\o{}rstad -> Bjoerstad
    gsub(/\\ss[{}]/,	"ss",	s) # F{\"o}{\ss}meier ->Foessmeier)

    gsub(/\\u\\i[{}]/,  "i",	s) # Kosovski{\u\i} -> Kosovskii

    gsub(/\\&/,		"",	s) # AT{\&T} -> ATT

    gsub(/'/,		"",	s) # Il'in -> Ilin
    gsub(/`/,		"",	s) # O`Hearn -> OHearn

    gsub(/[{}]/,	"",	s) # {R}oeck -> Roeck

    # Warn if we missed some.  These messages often indicate errors in the
    # input bibliography.
    if (s ~ /[^a-zA-Z-]/)
	warning("ERROR: incomplete accent removal <" names[n] "> -> <" s ">")

    # And then remove everything but letters and hyphen
    gsub(/[^a-zA-Z-]/,	"",	s)

    return (s)
}


function fix_author_editor(s, t)
{
    t = s

    # Change parentheses and periods to spaces
    gsub(/[().]/,		" ",	t)

    # Change explicit space to ordinary space
    gsub(/\\ /,			" ",	t)

    # Remove Jr-like modifiers
    gsub(/,[ ~]*[Jj]r.?/,	"",	t)
    gsub(/,[ ~]*[Ss]r.?/,	"",	t)
    gsub(/[ ~][Jj]r.?/,		"",	t)
    gsub(/[ ~][Ss]r.?/,		"",	t)
    gsub(/, *III/,		"",	t)
    gsub(/, *IV/,		"",	t)
    gsub(/, *II/,		"",	t)

    return (t)
}

function is_tag_in_use(tag, in_use)
{
    in_use = (tolower(tag) in in_use_tag) && \
	(FILENAME != in_use_file[tolower(tag)])
    print_debug("is_tag_in_use(" tag ") -> " in_use)
    return (in_use)
}


function journal_string()
{
    journal = value($0)
}


function make_suffixed_tag(tag)
{				# add a suffix to make a unique tag
    for (k = 1; k <= 26; ++k) # try suffixes "a", "b", ..., "z"
    {
	suffixed_new_tag = tag substr("abcdefghijklmnopqrstuvwxyz",k,1)
	if (suffixed_new_tag in new_to_old)
	    continue
	else if (is_tag_in_use(suffixed_new_tag))
	    continue
	else
	    return (suffixed_new_tag)
    }
    # exhausted "a" "b" ... "z", so (tail) recursively add suffixes
    return (make_suffixed_tag(tag "a"))
}


function make_ignore_list()
{
    # List of words to ignore in forming citation tags.  The initial
    # list was extracted from the bibindex badwords list, and covers
    # a few European languages as well as English.
    ignore["a"]         = 1
    ignore["ab"]        = 1
    ignore["aber"]      = 1
    ignore["als"]       = 1
    ignore["an"]        = 1
    ignore["and"]       = 1
    ignore["are"]       = 1
    ignore["as"]        = 1
    ignore["auf"]       = 1
    ignore["aus"]       = 1
    ignore["az"]        = 1
    ignore["bei"]       = 1
    ignore["bir"]       = 1
    ignore["but"]       = 1
    ignore["da"]        = 1
    ignore["das"]       = 1
    ignore["dat"]       = 1
    ignore["de"]        = 1
    ignore["dei"]       = 1
    ignore["dem"]       = 1
    ignore["den"]       = 1
    ignore["der"]       = 1
    ignore["des"]       = 1
    ignore["det"]       = 1
    ignore["di"]        = 1
    ignore["die"]       = 1
    ignore["dos"]       = 1
    ignore["e"]         = 1
    ignore["een"]       = 1
    ignore["eene"]      = 1
    ignore["egy"]       = 1
    ignore["ei"]        = 1
    ignore["ein"]       = 1
    ignore["eine"]      = 1
    ignore["einen"]     = 1
    ignore["einer"]     = 1
    ignore["eines"]     = 1
    ignore["eit"]       = 1
    ignore["el"]        = 1
    ignore["en"]        = 1
    ignore["er"]        = 1
    ignore["es"]        = 1
    ignore["et"]        = 1
    ignore["ett"]       = 1
    ignore["eyn"]       = 1
    ignore["eyne"]      = 1
    ignore["for"]       = 1
    ignore["from"]      = 1
    ignore["fuer"]      = 1
    ignore["fur"]       = 1
    ignore["gl"]        = 1
    ignore["gli"]       = 1
    ignore["ha"]        = 1
    ignore["haben"]     = 1
    ignore["had"]       = 1
    ignore["hai"]       = 1
    ignore["has"]       = 1
    ignore["hat"]       = 1
    ignore["have"]      = 1
    ignore["he"]        = 1
    ignore["heis"]      = 1
    ignore["hen"]       = 1
    ignore["hena"]      = 1
    ignore["henas"]     = 1
    ignore["het"]       = 1
    ignore["hin"]       = 1
    ignore["hinar"]     = 1
    ignore["hinir"]     = 1
    ignore["hinn"]      = 1
    ignore["hith"]      = 1
    ignore["ho"]        = 1
    ignore["hoi"]       = 1
    ignore["i"]         = 1
    ignore["il"]        = 1
    ignore["in"]        = 1
    ignore["ist"]       = 1
    ignore["ka"]        = 1
    ignore["ke"]        = 1
    ignore["l"]         = 1
    ignore["la"]        = 1
    ignore["las"]       = 1
    ignore["le"]        = 1
    ignore["les"]       = 1
    ignore["lo"]        = 1
    ignore["los"]       = 1
    ignore["mia"]       = 1
    ignore["mit"]       = 1
    ignore["n"]         = 1
    ignore["na"]        = 1
    ignore["nji"]       = 1
    ignore["not"]       = 1
    ignore["o"]         = 1
    ignore["oder"]      = 1
    ignore["of"]        = 1
    ignore["on"]        = 1
    ignore["or"]        = 1
    ignore["os"]        = 1
    ignore["others"]    = 1
    ignore["s"]         = 1
    ignore["sie"]       = 1
    ignore["sind"]      = 1
    ignore["so"]        = 1
    ignore["t"]         = 1
    ignore["ta"]        = 1
    ignore["the"]       = 1
    ignore["to"]        = 1
    ignore["um"]        = 1
    ignore["uma"]       = 1
    ignore["un"]        = 1
    ignore["una"]       = 1
    ignore["und"]       = 1
    ignore["une"]       = 1
    ignore["uno"]       = 1
    ignore["unter"]     = 1
    ignore["von"]       = 1
    ignore["with"]      = 1
    ignore["y"]         = 1
    ignore["yr"]        = 1

#   Additional words added later
    ignore["also"]      = 1
    ignore["any"]       = 1
    ignore["away"]      = 1
    ignore["by"]        = 1
    ignore["cum"]       = 1
    ignore["dans"]      = 1
    ignore["down"]      = 1
    ignore["into"]      = 1
    ignore["its"]       = 1
    ignore["off"]       = 1
    ignore["onto"]      = 1
    ignore["out"]       = 1
    ignore["over"]      = 1
    ignore["sur"]       = 1
    ignore["that"]      = 1
    ignore["these"]     = 1
    ignore["this"]      = 1
    ignore["those"]     = 1
    ignore["unto"]      = 1
    ignore["up"]        = 1
    ignore["via"]       = 1
    ignore["without"]   = 1
    ignore["zu"]        = 1
    ignore["zum"]       = 1
    ignore["zur"]       = 1
}


function make_unique_new_tag()
{
    # we have two duplicate tags, so find replacements for both, so e.g.
    # Smith:ABC1980 becomes Smith:ABC1980a and Smith:ABC1980b
    if (new_to_old[new_tag])	# then new_tag still in use
    {
        tag_1 = make_suffixed_tag(new_tag)
	replace_tag(new_tag,tag_1)
    }

    tag_2 = make_suffixed_tag(new_tag)
    new_to_old[tag_2] = new_to_old[new_tag]
    record_tag_use(FILENAME,new_to_old[tag_2],tag_2)

    print_debug("make_unique_new_tag: new_tag <" new_tag "> tag_1 <" \
	tag_1 "> tag_2 <" tag_2 ">")

    new_tag = tag_2
}


function number_string()
{
    number = value($0)
}


function pages_string()
{
    n = split(value($0),pages,"--")
}


function print_debug(s)
{
#    print "DEBUG: " FILENAME ":" FNR "\t[" old_tag "] " s >"/dev/tty"
    return 0 # dummy statement: gawk will not accept an empty function body
}


function print_substitution_line(old,new, k,delimiters,d)
{
    # sortpipe = "sort -b -f"		# to sort by old tags
    sortpipe = "sort -b -f +1 -2" 	# to sort by new tags

    printf("%s%*s%s\n", old_tag, 32 - length(old_tag), " ",
	old_to_new[old_tag]) | sortpipe
}


function record_tag_use(file,old,new)
{
    # Because BibTeX ignores letter case in tags and entry names, we
    # track tag usage by lowercased indexes into in_use_tag[].  However,
    # we preserve letter case of indexes into old_to_new[] so that
    # more readable mixed case tags (e.g. Smith:ABC1980) can be
    # supported.

    if (new)
	old_to_new[old] = new
    in_use_tag[tolower(old)] = 1
    in_use_file[tolower(old)] = file
}


function replace_tag(new,newer)
{
    # Copy original mapping to newer tag, and delete original.
    # However, we retain the old_to_new[] and in_use_tag[] status,
    # because they may refer to tags defined in other bibliography
    # files that we are not processing on this run, and anyway,
    # if we have just produced Smith:1980:ABCa, Smith:1980:ABCb, and
    # Smith:1980:ABCc, we don't want to free Smith:1980:ABC for later
    # use.
    new_to_old[newer] = new_to_old[new]
    new_to_old[new] = ""
    record_tag_use(FILENAME,new_to_old[newer],newer)
}


# Eventually, we may want to prepare more customized tags, but for
# now, most types use the book tag, Author:ABCyyyy, based on the title
# and four digits of the year.  In most cases, this still generates a
# unique tag, and anyway, in end_bibliography(), we ensure that unique
# tags are generated by adding letter suffixes where needed.

function tag_Article()
{
    return (tag_Book())
}


function tag_Book()
{
    print_debug("old_tag = <" old_tag "> Last name = <" lastname \
	"> ttlabb = <" title_abbrev "> year = <" year ">")

    if (lastname && title_abbrev && year)
	return lastname ":" year ":" title_abbrev
    else
	return ""
}

function tag_Booklet()
{
    return (tag_Book())
}


function tag_DEAthesis()
{
    return (tag_Book())
}


function tag_InBook()
{
    return (tag_Book())
}


function tag_InCollection()
{
    return (tag_Book())
}


function tag_InProceedings()
{
    return (tag_Book())
}


function tag_Manual()
{
    return (tag_Book())
}


function tag_MastersThesis()
{
    return (tag_Book())
}


function tag_Misc()
{
    return (tag_Book())
}


function tag_Periodical()
{
    return (tag_Book())
}


function tag_PhdThesis()
{
    return (tag_Book())
}


function tag_Proceedings()
{
    return (tag_Book())
}


function tag_TechReport()
{
    return (tag_Book())
}


function tag_Unpublished()
{
    return (tag_Book())
}


function title_string()
{
    # Save a 1- to 3-letter title_abbrev for use in making a citation
    # tag.  Braces are discarded, and only words beginning with a
    # letter are candidates, so that a title like
    # "${L}^{2}$ error bounds for the {R}ayleigh-{R}itz-{G}alerkin method"
    # will reduce to EBR.
    s = tolower(value($0))	# need uniform case for ignore[] lookup
    gsub(/[{}]/,"",s)		# strip braces
    gsub(/[---().,;:!?]/," ",s)	# change hyphens and punctuation to spaces
    gsub(/[\\]/," ",s)		# change backslash to space (so \TeX -> TeX)
    n = split(s,titlewords)
    title_abbrev = ""
    for (k = 1; (k <= n) && (length(title_abbrev) < 3); ++k)
    {
	print_debug("considering <" titlewords[k] ">")
	if ((!(titlewords[k] in ignore)) && (titlewords[k] ~ /^[A-Za-z]/))
	{
	    print_debug("using <" titlewords[k] ">")
	    title_abbrev = title_abbrev toupper(substr(titlewords[k],1,1))
	}
    }
}


function value(s)
{
    # Return the value string, EXCLUDING surrounding quotes, of a
    # `key = "value",' or `key = abbrev' pair
    k1 = index(s,"\"")
    k2 = index(s,"\",")
    if ((k2 == 0) && (k1 > 0))	# recognized unclosed strings
	k2 = length($0) + 1
    if (k2 > k1)
	return (substr(s,k1+1,k2 - (k1 + 1)))
    k1 = index(s,"= ")
    k2 = index(s,",")
    return (substr(s,k1+2,k2 - (k1 + 2)))
}


function volume_string()
{
    volume = value($0)
}


function warning(s)
{
    print FILENAME ":" FNR "\t[" old_tag "]\t" s >"/dev/tty"
}


function year_string()
{
    year = value($0)
    year = substr(year,1,4)
    if (year ~ /1[0-9][0-9][0-9]/)
	return
    else if (year ~ /20[0-9][0-9]/)
	return
    else
    {
	warning("year <" year "> is out of range [1000..2099]: " \
	    "not acceptable for a citation tag")
 	year = "19xx"	# xx, not ??, to avoid interfering with awk patterns
    }
}
