### ====================================================================
###  @Awk-file{
###     author          = "Nelson H. F. Beebe",
###     version         = "0.02",
###     date            = "16 October 1996",
###     time            = "11:49:00 MDT",
###     filename        = "biblabel.awk",
###     address         = "Center for Scientific Computing
###                        Department of Mathematics
###                        University of Utah
###                        Salt Lake City, UT 84112
###                        USA",
###     telephone       = "+1 801 581 5254",
###     FAX             = "+1 801 581 4148",
###     checksum        = "23784 961 3712 30406",
###     email           = "beebe@math.utah.edu (Internet)",
###     codetable       = "ISO/ASCII",
###     keywords        = "bibliography, BibTeX, citation label, LaTeX,
###                        TeX",
###     supported       = "yes",
###     docstring       = "This program reads one or more bibliography
###                        files, and prepares a list of unique
###                        standardized citation labels for selected
###                        document types.  The citation label is formed
###                        by these rules, easily applicable by a
###                        human, or by a computer program like this
###                        one:
###
###                        (1) Take the first author's last name,
###                            dropping apostrophes, Jr/Sr/generation
###                            numbers, and eliminating accents
###                            (e.g. J{\"a}nsch -> Jaensch, and
###                            Jind\v{r}ich -> Jindrich), using
###                            multi-letter transliterations if that is
###                            conventional.  Hyphenated names, like
###                            Baeza-Yates, are preserved in full.
###                        (2) Append a colon.
###                        (3) Append the four-digit year of publication
###                        (4) Append another colon.
###                        (5) Pick the first 3 important words in the
###                            title that begin with a letter, excluding
###                            articles and prepositions and TeX math
###                            mode (e.g. ``On ${C}^1$ interpolating
###                            hierarchical spline bases'' reduces to
###                            IHS), and append that.  If there are
###                            fewer than 3 important words, then use a
###                            shorter string.
###                        (6) If the resulting label is already in use,
###                            add a letter a, b, c, ... to make it
###                            unique.
###
###                        This will produce a label like
###                        Smith:1994:ABC.
###
###                        The reason for including a four-digit year is
###                        that the millenium change is near, and we
###                        expect bibliographies to be in use for many
###                        years hence.  Also, some bibliographies may
###                        be historical, with entries dating back
###                        hundreds of years.  Using a four-digit year
###                        will keep sorts of otherwise identical keys
###                        in chronological order, and putting the year
###                        before the key derived from the title will
###                        facilitate sorting by year.
###
###                        Because any change in citation labels must be
###                        accompanied by a change in citations in all
###                        documents that use the bibliography, it is
###                        not sufficient to just produce a new
###                        bibliography file with changed labels.
###
###                        Instead, the output is a file containing old
###                        and new labels, one per line, suitable for
###                        input to the companion citesub program
###                        (either awk or C versions).
###
###                        To avoid confusion between labels with common
###                        prefixes, such as Smith80 and Smith80a,
###                        citesub will check for leading context of a
###                        left brace, quote, comma, whitespace, or
###                        beginning of line and trailing context of a
###                        right brace, comma, quote, percent,
###                        whitespace, or end of line so as to match
###                        these styles:
###
###                             @Book{Smith:1980:ABC,
###                             crossref = <quote>Smith:1980:ABC<quote>,
###                             crossref = {Smith:1980:ABC},
###                             \cite{Smith:1980:ABC}
###                             \cite{Smith:1980:ABC,Jones:1994:DEF}
###                             \cite{%
###                                     Smith:1980:ABC,%
###                                     Jones:1994:DEF%
###                             }
###
###                        The Ignore[] list initialized in
###                        make_ignore_list() contains words which are
###                        to be ignored in preparation of the
###                        alphabetic string from the title words.  It
###                        will be augmented at run-time from two files:
###                        biblabel.ign (intended to be a generic
###                        enhancement), and a file assigned to IGNORE
###                        on the command line.
###
###                        Created labels are guaranteed to be unique
###                        within the input files provided on the
###                        command line.  However, in a larger project,
###                        one may wish to exclude labels that are
###                        already in use in other bibliographies.  To
###                        provide for this, the USED variable can be
###                        set on the command line to define the name of
###                        a file of labels that are already in use.  It
###                        will be augmented by labels from a generic
###                        file biblabel.use.  Lines in these file
###                        consist of pairs of filenames and citation
###                        labels.
###
###                        When a label is found in use, and the current
###                        file matches the in-use label filename, the
###                        label is considered to be unused; otherwise,
###                        repeated runs through this program would keep
###                        changing already-assigned labels.
###
###                        Usage:
###                             nawk -f biblabel.awk
###                                     [-v IGNORE=ignore-file]
###                                     [-v USED=used-labels-file]
###                                     mybib.bib >mybib.sub
###
###                        Examine the output substitutions to decide
###                        whether the generated labels are acceptable,
###                        and make any changes by hand.
###
###                        Tentatively apply the label substitutions and
###                        compare the input and output files:
###
###                        citesub -f mybib.sub mybib.bib >mybib.bib.tmp
###                        diff mybib.bib mybib.bib.tmp
###
###                        citesub -f mybib.sub myfile.ltx >myfile.ltx.tmp
###                        diff myfile.ltx myfile.ltx.tmp
###
###                        If the citesub substitutions are acceptable,
###                        then:
###
###                        mv mybib.bib.tmp mybib.tmp
###                        mv myfile.ltx.tmp myfile.ltx
###
###                        The checksum field above contains a CRC-16
###                        checksum as the first value, followed by the
###                        equivalent of the standard UNIX wc (word
###                        count) utility output of lines, words, and
###                        characters.  This is produced by Robert
###                        Solovay's checksum utility.",
###  }
### ====================================================================

BEGIN 					{ begin_bibliography() }

/^@[A-Za-z][A-Za-z0-9_-]*/		{ begin_entry(); next }

/^ *author *= *"/			{ author_string(); next }

/^ *editor *= *"/			{ editor_string(); next }

/^ *journal *= *j-.*, *$/		{ journal_string(); next }

/^ *number *= *"/			{ number_string(); next }

/^ *pages *= *"/			{ pages_string(); next }

/^ *title *= *"/			{ title_string(); next }

					# Use the booktitle only when the title
					# has not been seen yet, and if a title
					# is subsequently seen, we will override
					# Title_abbrev from booktitle by a new
					# Title_abbrev from title.
/^ *booktitle *= *"/			{ if (Title_abbrev == "") title_string(); next }

/^ *volume *= *"/			{ volume_string(); next }

/^ *year *= *"/				{ year_string(); next }

/^}/ 					{ end_entry(); next }

					{ next }

END					{ end_bibliography() }

#=======================================================================

function author_string( authors,n,names,s)
{
    s = fix_author_editor(value($0))
    ## print_debug("author_string(): authors = <" s ">")
    n = split(s,authors," and ")
    ## print_debug("author_string(): First author = <" authors[1] ">")
    n = split(authors[1],names," ")
    Lastname = names[n]		# last name of first author

    Lastname = fix_accents(Lastname)
}


function begin_bibliography()
{
    make_ignore_list()

    extend_ignore_list("biblabel.ign")
    if (IGNORE)
	extend_ignore_list(IGNORE)

    extend_used_list("biblabel.use")
    if (USED)
	extend_used_list(USED)
}



function begin_entry()
{
    Lastname = ""
    New_tag = ""
    Old_tag = entry_tag($0)
    Pages[1] = ""
    Title_abbrev = ""
    Type = entry_type($0)
    Year = "19xx"	# default year so we can always make a citation tag
}


function delete_substring(s,rstart,rlength)
{
    return (substr(s,1,rstart-1) substr(s,rstart+rlength))
}


function editor_string( editors,n,names,s)
{
    if (Lastname)	# don't set Lastname if already set from authors
	return
    s = fix_author_editor(value($0))
    ## print_debug("editor_string(): editors = <" s ">")
    n = split(s,editors," and ")
    ## print_debug("editor_string(): First editor = <" editors[1] ">")
    n = split(editors[1],names," ")
    Lastname = names[n]		# last name of first author
    Lastname = fix_accents(Lastname)
}


function end_bibliography()
{
    for (Old_tag in Old_to_new)
    {
	# NB: we print ALL tags here, even if (Old_tag == Old_to_new[Old_tag]),
	# because we later extract the new tags from the *.sub files to
	# prepare the biblabel.use file
	if (Old_tag && Old_to_new[Old_tag])
	    print_substitution_line(Old_tag,Old_to_new[Old_tag])
    }
}


function end_entry()
{
    if (Type == "Article") 		New_tag = tag_Article()
    else if (Type == "Book")		New_tag = tag_Book()
    else if (Type == "Booklet")		New_tag = tag_Booklet()
    else if (Type == "DEAthesis")	New_tag = tag_DEAthesis()
    else if (Type == "InBook")		New_tag = tag_InBook()
    else if (Type == "InCollection")	New_tag = tag_InCollection()
    else if (Type == "InProceedings")	New_tag = tag_InProceedings()
    else if (Type == "Manual")		New_tag = tag_Manual()
    else if (Type == "MastersThesis")	New_tag = tag_MastersThesis()
    else if (Type == "Misc")		New_tag = tag_Misc()
    else if (Type == "Periodical")	New_tag = tag_Periodical()
    else if (Type == "PhdThesis")	New_tag = tag_PhdThesis()
    else if (Type == "Proceedings")	New_tag = tag_Proceedings()
    else if (Type == "TechReport")	New_tag = tag_TechReport()
    else if (Type == "Unpublished")	New_tag = tag_Unpublished()
    else				New_tag = ""

    ## print_debug("end_entry(): Old_tag = <" Old_tag "> Type = <" Type "> New_tag = <" New_tag ">")
    if (is_tag_in_use(Old_tag))		# duplicate tag in input bibliography
	warning("ERROR: duplicate tag <" Old_tag "> in input file(s)")
    else
    {
	if (New_tag)			# then generated a valid new tag
	{
	    if ((New_tag in New_to_old) || \
		is_tag_in_use(New_tag)) # then duplicate tag
		New_tag = make_unique_new_tag(New_tag)
	    New_to_old[New_tag] = Old_tag
	    record_tag_use(FILENAME,Old_tag,New_tag)
	}
	else			# remember old tags to check for dups
	    record_tag_use(FILENAME,Old_tag,"")
    }
    Type = ""			# forget current type and tag
    Old_tag = ""		# so as to catch unbalanced braces
}


function entry_tag(s, k1,k2)
{
    # Return the citation tag from a `@EntryType{tag,' line
    k1 = index(s,"{") + 1
    k2 = index(s,",")
    return (substr(s,k1,k2-k1))
}


function entry_type(s)
{
    return (substr(s,2,index(s,"{")-2))
}


function extend_ignore_list(filename, word)
{
    if (file_exists(filename))
    {
	while (getline word <filename)
	{
	    gsub(/[^a-zA-Z]/,"",word)	# discard all but letters
	    word = tolower(word)	# and convert to lower case
	    Ignore[word] = 1
	    print "DEBUG: ignoring word <" word ">"
	}
    }
}


function extend_used_list(filename, line,old_file,old_tag,parts)
{
    if (file_exists(filename))
    {
	while (getline line <filename)
	{
	    split(line,parts,"\t")
	    old_file = parts[1]
	    old_tag = parts[2]
	    gsub(/^[ \t]*/,"",old_tag)	# discard leading and
	    gsub(/[ \t]$/,"",old_tag) 	# trailing space
	    record_tag_use(old_file,old_tag,"")
	    ## print_debug("extend_used_list(): excluding old_tag = <" old_tag "> from file = [" old_file "]")
	}
    }
}


function file_exists(filename)
{
    # Unfortunately, getline in both nawk and gawk will hang if its
    # file does not exist, so we need to test for file existence by
    # invoking a shell command, sigh...

    return (system("cat " filename " >/dev/null 2>&1") == 0)
}


function fix_accents(Lastname, s)
{
    s = Lastname

    # Remove accents from last name, because we need a control-sequence
    # free tag.  The order of some of these is important, so do NOT
    # rearrange them!
    # Eventually, an algorithmic reduction should be applied here instead
    # of all of these substitutions.

    # eliminate acute, grave, circumflex, macron, tilde accents
    gsub(/\\[.'`^=~]/,	"",	s)

    gsub(/\\[uvHtcdb]{/,"",	s) # eliminate most other accents

    gsub(/~/,		" ",	s) # De~Raedt -> De Raedt

    gsub(/{\\i}/,	"i",	s) # dotless i -> i
    gsub(/{\\j}/,	"j",	s) # dotless j -> j
    gsub(/{\\l}/,	"l",	s) # Ko{\\l}os -> Kolos

    gsub(/\\"A/,	"Ae",	s) #
    gsub(/\\"O/,	"Oe",	s) #
    gsub(/\\"U/,	"Ue",	s) #
    gsub(/\\"\\i/,	"i",	s) # Vorono{\"\i} -> Voronoi
    gsub(/\\"a/,	"ae",	s) # J{\"a}nsch -> Jaensch
    gsub(/\\"e/,	"e",	s) # Eli{\"e}ns -> Eliens
    gsub(/\\"i/,	"i",	s) # Bsa{\"i}es -> Bsaies
    gsub(/\\"o/,	"oe",	s) # B{\"o}rgers -> Boergers
    gsub(/\\"u/,	"ue",	s) # R{\"u}de -> Ruede
    gsub(/\\"y/,	"y",	s) # Delabbe{\"y} ->Delabbey
    gsub(/\\"{A}/,	"ae",	s) #
    gsub(/\\"{O}/,	"oe",	s) #
    gsub(/\\"{U}/,	"ue",	s) #
    gsub(/\\"{a}/,	"ae",	s) # J{\"{a}}nsch -> Jaensch
    gsub(/\\"{e}/,	"e",	s) # Eli{\"{e}}ns -> Eliens
    gsub(/\\"{i}/,	"i",	s) # Bsa{\"{i}}es -> Bsaies
    gsub(/\\"{o}/,	"oe",	s) # B{\"{o}}rgers -> Boergers
    gsub(/\\"{u}/,	"ue",	s) # H{\"{u}}bner -> Huebner

    gsub(/\\AA[{}]/,	"Aa",	s) # {\AA}rhus -> Aarhus
    gsub(/\\AE[{}]/,	"Ae",	s) # {\AE}ro -> Aero
    gsub(/\\OE[{}]/,	"Oe",	s) # {\OE}rsted -> Oersted
    gsub(/\\aa[{}]/,	"aa",	s) # Ring\aa -> Ringaa
    gsub(/\\ae[{}]/,	"ae",	s) # D{\ae}hlen -> Daehlen
    gsub(/\\oe[{}]/,	"ae",	s) # St{\oe}ren -> Stoeren
    gsub(/\\o[{}]/,	"oe",	s) # Bj\o{}rstad -> Bjoerstad
    gsub(/\\ss[{}]/,	"ss",	s) # F{\"o}{\ss}meier ->Foessmeier)

    gsub(/\\u\\i[{}]/,  "i",	s) # Kosovski{\u\i} -> Kosovskii

    gsub(/\\&/,		"",	s) # AT{\&T} -> ATT

    gsub(/'/,		"",	s) # Il'in -> Ilin
    gsub(/`/,		"",	s) # O`Hearn -> OHearn

    gsub(/[{}]/,	"",	s) # {R}oeck -> Roeck

    # Warn if we missed some.  These messages often indicate errors in the
    # input bibliography.
    if (s ~ /[^a-zA-Z-]/)
	warning("ERROR: incomplete accent removal <" Lastname "> -> <" s ">")

    # And then remove everything but letters and hyphen
    gsub(/[^a-zA-Z-]/,	"",	s)

    return (s)
}


function fix_author_editor(s, k,n,name,t)
{
    t = s

    # Change parentheses and periods to spaces
    gsub(/[().]/,		" ",	t)

    # Change explicit space to ordinary space
    gsub(/\\ /,			" ",	t)

    # Remove Jr-like modifiers.
    # Problem cases:
    #    "John Smith Sr and A. Srinivasan"
    #    "A. Srinivasan and John Smith Sr"
    #    "A. Srinivasan Sr and A. Srinivasan Jr"
    #    "Shu-T'ien Xi"
    # These cannot be handled reliably by sub() and gsub(), because
    # awk's regular expression matching has no provision for
    # specification of trailing context.  They can be dealt with if we
    # first split the field into names of individuals, check for
    # trailing context by a separate match() call, and then delete
    # the matched string, minus the trailing context.
    n = split(t, name, " and ")
    for (k = 1; k <= n; ++k)
    {
	# handle upper-case Roman numerals
	if (match(name[k],/,?[ ~]+[IVXLCDM]+[^A-Za-z]/))
	    name[k] = delete_substring(name[k],RSTART,RLENGTH-1)
	else if (match(name[k],/,?[ ~]+[IVXLCDM]+$/))
	    name[k] = delete_substring(name[k],RSTART,RLENGTH)

	# handle Jr
	if (match(name[k],/,?[ ~]+[Jj]r[^A-Za-z]/))
	    name[k] = delete_substring(name[k],RSTART,RLENGTH-1)
	else if (match(name[k],/,?[ ~]+[Jj]r$/))
	    name[k] = delete_substring(name[k],RSTART,RLENGTH)

	# handle Sr
	if (match(name[k],/,?[ ~]*[Ss]r[^A-Za-z]/))
	    name[k] = delete_substring(name[k],RSTART,RLENGTH-1)
	else if (match(name[k],/,?[ ~]+[Ss]r$/))
	    name[k] = delete_substring(name[k],RSTART,RLENGTH)
    }
    t = name[1]
    for (k = 2; k <= n; ++k)
	t = t " and " name[k]
    gsub(/ +/," ",t)
    return (t)
}


function is_tag_in_use(tag, in_use)
{
    # Version 0.01 had
    #
    #  in_use = (tolower(tag) in In_use_tag) && (FILENAME != In_use_file[tolower(tag)])
    #
    # The second test for filename mismatched was based on the
    # assumption that the input file contained no duplicate labels, so
    # there was no need to check.  The result of this was that (a) no
    # duplicate label warning was ever issued, and (b) long label
    # suffixes (e.g. aaaag) were generated for duplicate labels.
    # At Version 0.02, this was changed to test only for membership in
    # the In_use_tag[] array.

    in_use = (tolower(tag) in In_use_tag)
    ## print_debug("is_tag_in_use(" tag ") -> " in_use " FILENAME=[" FILENAME "] In_use_file=[" In_use_file[tolower(tag)] "]")
    return (in_use)
}


function journal_string()
{
    Journal = value($0)
}


function make_suffixed_tag(tag, k,suffixed_new_tag)
{				# add a suffix to make a unique tag
    for (k = 1; k <= 26; ++k) # try suffixes "a", "b", ..., "z"
    {
	suffixed_new_tag = tag substr("abcdefghijklmnopqrstuvwxyz",k,1)
	if (suffixed_new_tag in New_to_old)
	{
	    ## print_debug("make_suffixed_tag(" tag "): [" suffixed_new_tag "] in New_to_old: value=[" New_to_old[suffixed_new_tag] "]")
	    continue
	}
	else if (is_tag_in_use(suffixed_new_tag))
	{
	    ## print_debug("make_suffixed_tag(" tag "): is_tag_in_use(" suffixed_new_tag ") -> 1")
	    continue
	}
	else
	    return (suffixed_new_tag)
    }
    # exhausted "a" "b" ... "z", so (tail) recursively add suffixes
    return (make_suffixed_tag(tag "a"))
}


function make_ignore_list()
{
    # List of words to Ignore in forming citation tags.  The initial
    # list was extracted from the bibindex badwords list, and covers
    # a few European languages as well as English.
    Ignore["a"]         = 1
    Ignore["ab"]        = 1
    Ignore["aber"]      = 1
    Ignore["als"]       = 1
    Ignore["an"]        = 1
    Ignore["and"]       = 1
    Ignore["are"]       = 1
    Ignore["as"]        = 1
    Ignore["auf"]       = 1
    Ignore["aus"]       = 1
    Ignore["az"]        = 1
    Ignore["bei"]       = 1
    Ignore["bir"]       = 1
    Ignore["but"]       = 1
    Ignore["da"]        = 1
    Ignore["das"]       = 1
    Ignore["dat"]       = 1
    Ignore["de"]        = 1
    Ignore["dei"]       = 1
    Ignore["dem"]       = 1
    Ignore["den"]       = 1
    Ignore["der"]       = 1
    Ignore["des"]       = 1
    Ignore["det"]       = 1
    Ignore["di"]        = 1
    Ignore["die"]       = 1
    Ignore["dos"]       = 1
    Ignore["e"]         = 1
    Ignore["een"]       = 1
    Ignore["eene"]      = 1
    Ignore["egy"]       = 1
    Ignore["ei"]        = 1
    Ignore["ein"]       = 1
    Ignore["eine"]      = 1
    Ignore["einen"]     = 1
    Ignore["einer"]     = 1
    Ignore["eines"]     = 1
    Ignore["eit"]       = 1
    Ignore["el"]        = 1
    Ignore["en"]        = 1
    Ignore["er"]        = 1
    Ignore["es"]        = 1
    Ignore["et"]        = 1
    Ignore["ett"]       = 1
    Ignore["eyn"]       = 1
    Ignore["eyne"]      = 1
    Ignore["for"]       = 1
    Ignore["from"]      = 1
    Ignore["fuer"]      = 1
    Ignore["fur"]       = 1
    Ignore["gl"]        = 1
    Ignore["gli"]       = 1
    Ignore["ha"]        = 1
    Ignore["haben"]     = 1
    Ignore["had"]       = 1
    Ignore["hai"]       = 1
    Ignore["has"]       = 1
    Ignore["hat"]       = 1
    Ignore["have"]      = 1
    Ignore["he"]        = 1
    Ignore["heis"]      = 1
    Ignore["hen"]       = 1
    Ignore["hena"]      = 1
    Ignore["henas"]     = 1
    Ignore["het"]       = 1
    Ignore["hin"]       = 1
    Ignore["hinar"]     = 1
    Ignore["hinir"]     = 1
    Ignore["hinn"]      = 1
    Ignore["hith"]      = 1
    Ignore["ho"]        = 1
    Ignore["hoi"]       = 1
    Ignore["i"]         = 1
    Ignore["il"]        = 1
    Ignore["in"]        = 1
    Ignore["ist"]       = 1
    Ignore["ka"]        = 1
    Ignore["ke"]        = 1
    Ignore["l"]         = 1
    Ignore["la"]        = 1
    Ignore["las"]       = 1
    Ignore["le"]        = 1
    Ignore["les"]       = 1
    Ignore["lo"]        = 1
    Ignore["los"]       = 1
    Ignore["mia"]       = 1
    Ignore["mit"]       = 1
    Ignore["n"]         = 1
    Ignore["na"]        = 1
    Ignore["nji"]       = 1
    Ignore["not"]       = 1
    Ignore["o"]         = 1
    Ignore["oder"]      = 1
    Ignore["of"]        = 1
    Ignore["on"]        = 1
    Ignore["or"]        = 1
    Ignore["os"]        = 1
    Ignore["others"]    = 1
    Ignore["s"]         = 1
    Ignore["sie"]       = 1
    Ignore["sind"]      = 1
    Ignore["so"]        = 1
    Ignore["t"]         = 1
    Ignore["ta"]        = 1
    Ignore["the"]       = 1
    Ignore["to"]        = 1
    Ignore["um"]        = 1
    Ignore["uma"]       = 1
    Ignore["un"]        = 1
    Ignore["una"]       = 1
    Ignore["und"]       = 1
    Ignore["une"]       = 1
    Ignore["uno"]       = 1
    Ignore["unter"]     = 1
    Ignore["von"]       = 1
    Ignore["with"]      = 1
    Ignore["y"]         = 1
    Ignore["yr"]        = 1

#   Additional words added later
    Ignore["also"]      = 1
    Ignore["any"]       = 1
    Ignore["away"]      = 1
    Ignore["be"]        = 1
    Ignore["by"]        = 1
    Ignore["cum"]       = 1
    Ignore["dans"]      = 1
    Ignore["down"]      = 1
    Ignore["into"]      = 1
    Ignore["is"]        = 1
    Ignore["its"]       = 1
    Ignore["off"]       = 1
    Ignore["onto"]      = 1
    Ignore["out"]       = 1
    Ignore["over"]      = 1
    Ignore["sur"]       = 1
    Ignore["that"]      = 1
    Ignore["these"]     = 1
    Ignore["this"]      = 1
    Ignore["those"]     = 1
    Ignore["unto"]      = 1
    Ignore["up"]        = 1
    Ignore["via"]       = 1
    Ignore["without"]   = 1
    Ignore["zu"]        = 1
    Ignore["zum"]       = 1
    Ignore["zur"]       = 1
}


function make_unique_new_tag(new_tag, tag_1,tag_2)
{
    ## print_debug("make_unique_new_tag: new_tag <" new_tag ">")

    # we have two duplicate tags, so find replacements for both, so e.g.
    # Smith:ABC1980 becomes Smith:ABC1980a and Smith:ABC1980b
    if (New_to_old[new_tag])	# then new_tag still in use
    {
        tag_1 = make_suffixed_tag(new_tag)
	replace_tag(new_tag,tag_1)
    }

    tag_2 = make_suffixed_tag(new_tag)
    New_to_old[tag_2] = New_to_old[new_tag]
    # BUG: New_to_old[tag_2] might not exist yet:
    record_tag_use(FILENAME,New_to_old[tag_2],tag_2)

    ## print_debug("make_unique_new_tag: new_tag <" new_tag "> tag_1 <" tag_1 "> tag_2 <" tag_2 "> New_to_old[" tag_2 "] = <" New_to_old[tag_2] ">" )

    new_tag = tag_2
    return (new_tag)
}


function number_string()
{
    Number = value($0)
}


function pages_string()
{
    split(value($0),Pages,"--")
}


function print_debug(s)
{
    print "DEBUG: " FILENAME ":" FNR "\t[" Old_tag "] " s >"/dev/stderr"
    return 0 # dummy statement: gawk will not accept an empty function body
}


function print_substitution_line(old,new, k,delimiters,d,sortpipe)
{
    # sortpipe = "sort -b -f"		# to sort by old tags
    sortpipe = "sort -b -f +1 -2" 	# to sort by new tags

    printf("%s%*s%s\n", Old_tag, 32 - length(Old_tag), " ",
	Old_to_new[Old_tag]) | sortpipe
}


function record_tag_use(file,old,new, t)
{
    # Because BibTeX Ignores letter case in tags and entry names, we
    # track tag usage by lowercased indexes into In_use_tag[].  However,
    # we preserve letter case of indexes into Old_to_new[] so that
    # more readable mixed case tags (e.g. Smith:ABC1980) can be
    # supported.

    if (new)
	Old_to_new[old] = new
    t = tolower(old)
    In_use_tag[t] = 1
    In_use_file[t] = file
    ## print_debug("record_tag_use(" file "," old "," new ")")
}


function replace_tag(new,newer)
{
    # Copy original mapping to newer tag, and delete original.
    # However, we retain the Old_to_new[] and In_use_tag[] status,
    # because they may refer to tags defined in other bibliography
    # files that we are not processing on this run, and anyway,
    # if we have just produced Smith:1980:ABCa, Smith:1980:ABCb, and
    # Smith:1980:ABCc, we don't want to free Smith:1980:ABC for later
    # use.
    New_to_old[newer] = New_to_old[new]
    New_to_old[new] = ""
    record_tag_use(FILENAME,New_to_old[newer],newer)
}


function strip_math(s, k,n,parts,t)
{
    t = s
    gsub(/[$][$]/,"$",t)	# reduce displaymath to ordinary math
    gsub(/\\$/," ",t)		# change \$ to space
    n = split(t,parts,"[$]")	# split into non-math and math alternates
    t = ""
    for (k = 1; k <= n; k += 2)
	t = t ((k > 1) ? " " : "") parts[k]
    return (t)
}


# Eventually, we may want to prepare more customized tags, but for
# now, most types use the book tag, Author:ABCyyyy, based on the title
# and four digits of the year.  In most cases, this still generates a
# unique tag, and anyway, in end_bibliography(), we ensure that unique
# tags are generated by adding letter suffixes where needed.

function tag_Article()
{
    return (tag_Book())
}


function tag_Book()
{
    ## print_debug("tag_Book(): Old_tag = <" Old_tag "> Last name = <" Lastname "> ttlabb = <" Title_abbrev "> Year = <" Year ">")

    if (Lastname && Title_abbrev && Year)
	return Lastname ":" Year ":" Title_abbrev # BibNet style: "Smith:1996:AMS"
    else
	return ""		# insufficient information to create a tag
}


function tag_Booklet()
{
    return (tag_Book())
}


function tag_DEAthesis()
{
    return (tag_Book())
}


function tag_InBook()
{
    return (tag_Book())
}


function tag_InCollection()
{
    return (tag_Book())
}


function tag_InProceedings()
{
    return (tag_Book())
}


function tag_Manual()
{
    return (tag_Book())
}


function tag_MastersThesis()
{
    return (tag_Book())
}


function tag_Misc()
{
    return (tag_Book())
}


function tag_Periodical()
{
    return (tag_Book())
}


function tag_PhdThesis()
{
    return (tag_Book())
}


function tag_Proceedings()
{
    return (tag_Book())
}


function tag_TechReport()
{
    return (tag_Book())
}


function tag_Unpublished()
{
    return (tag_Book())
}


function title_string( k,n,s,titlewords)
{
    # Save a 1- to 3-letter Title_abbrev for use in making a citation
    # tag.  Braces are discarded, and only words beginning with a
    # letter are candidates, so that a title like
    # "${L}^{2}$ error bounds for the {R}ayleigh-{R}itz-{G}alerkin method"
    # will reduce to EBR.
    s = tolower(value($0))	# need uniform case for Ignore[] lookup
    s = strip_math(s)		# discard math mode
    gsub(/[{}]/,"",s)		# strip braces
    gsub(/``/," ",s)		# change quotation marks to space
    gsub(/''/," ",s)
    gsub(/[---().,;:!?]/," ",s) # change hyphens and punctuation to spaces
    gsub("/"," ",s)
    gsub(/\\emdash/," ",s)
    gsub(/\\slash/," ",s)
    gsub(/[\\]/," ",s)		# change backslash to space (so \TeX -> TeX)
    n = split(s,titlewords)
    Title_abbrev = ""
    for (k = 1; (k <= n) && (length(Title_abbrev) < 3); ++k)
    {
	## print_debug("title_string(): considering <" titlewords[k] ">")
	if ((!(titlewords[k] in Ignore)) && (titlewords[k] ~ /^[A-Za-z]/))
	{
	    ## print_debug("title_string(): using <" titlewords[k] ">")
	    Title_abbrev = Title_abbrev toupper(substr(titlewords[k],1,1))
	}
    }
}


function value(s, k1,k2)
{
    # Return the value string, EXCLUDING surrounding quotes, of a
    # `key = "value",' or `key = abbrev' pair
    k1 = index(s,"\"")
    k2 = index(s,"\",")
    if ((k2 == 0) && (k1 > 0))	# recognized unclosed strings
	k2 = length($0) + 1
    if (k2 > k1)
	return (substr(s,k1+1,k2 - (k1 + 1)))
    k1 = index(s,"= ")
    k2 = index(s,",")
    return (substr(s,k1+2,k2 - (k1 + 2)))
}


function volume_string()
{
    Volume = value($0)
}


function warning(s)
{
    print FILENAME ":" FNR "\t[" Old_tag "]\t" s >"/dev/stderr"
}


function year_string()
{
    Year = value($0)
    Year = substr(Year,1,4)
    if (Year ~ /1[0-9][0-9][0-9]/)
	return
    else if (Year ~ /20[0-9][0-9]/)
	return
    else
    {
	warning("year <" Year "> is out of range [1000..2099]: " \
	    "not acceptable for a citation tag: using 19xx instead")
 	Year = "19xx"	# xx, not ??, to avoid interfering with awk patterns
    }
}
