#! /usr/local/bin/bash
### ====================================================================
###  @UNIX-shell-file{
###     author          = "Nelson H. F. Beebe",
###     version         = "0.04",
###     date            = "06 March 2001",
###     time            = "06:11:30 MST",
###     filename        = "biblabel.sh",
###     address         = "Center for Scientific Computing
###                        University of Utah
###                        Department of Mathematics, 322 INSCC
###                        155 S 1400 E RM 233
###                        Salt Lake City, UT 84112-0090
###                        USA",
###     telephone       = "+1 801 581 5254",
###     FAX             = "+1 801 585 1640, +1 801 581 4148",
###     URL             = "http://www.math.utah.edu/~beebe",
###     checksum        = "33446 191 1024 8333",
###     email           = "beebe@math.utah.edu, beebe@acm.org,
###                        beebe@computer.org, beebe@ieee.org
###                        (Internet)",
###     codetable       = "ISO/ASCII",
###     keywords        = "bibliography, BibTeX, citation label, LaTeX,
###                        TeX",
###     supported       = "yes",
###     docstring       = "This script runs the biblabel.awk program
###                        to generate a citation label substitution
###                        file suitable for input to citesub(1) to
###                        standardize citation labels to the form
###                        Author:year:abbrev.
###
###                        Usage:
###
###                        biblabel [ --? ]
###                                 [ --author ]
###                                 [ --copyright ]
###                                 [ --corporate-file input-corporate-words-to-ignore-file ]
###                                 [ --dump-corporate-file output-file ]
###                                 [ --dump-ignore-file output-file ]
###                                 [ --dump-label-file output-file ]
###                                 [ --help ]
###                                 [ --ignore-file input-words-to-ignore-file ]
###                                 [ --long-corporate-names ]
###                                 [ --used-file input-labels-in-use-file ]
###                                 [ --version ]
###                                 [ BibTeXfile(s) ] >output-citesub-file
###
###                        Options may begin with either one or two
###                        hyphens, and may be abbreviated to any unique
###                        prefix, unless documented otherwise.
###
###                        For compatibility with older versions, the
###                        --used-file option may be called --f.
###
###                        The --? and --help options produce a brief
###                        help message, and then execution immediately
###                        terminates with a success return code (0).
###
###                        The --author option produces an author
###                        credit, and then execution immediately
###                        terminates with a success return code (0).
###
###                        The --copyright option produces a copyright
###                        statement, and then execution immediately
###                        terminates with a success return code (0).
###
###                        The --corporate-file option is followed by the
###                        name of a file containing words (one or more
###                        per line) to be added to the internal list of
###                        words that are ignored when forming corporate
###                        name abbreviations.  Multiple --corporate-file
###                        options may be specified.
###
###                        The --dump-ignore-file option produces a
###                        dump of words in the internal ignore list
###                        on the specified output file.  Multiple
###                        --dump-ignore-file options may be
###                        specified.
###
###                        The --dump-corporate-file option produces a
###                        dump of words in the internal corporate
###                        names ignore list on the specified output
###                        file.  Multiple --dump-corporate-file
###                        options may be specified.
###
###                        The --ignore-file option is followed by the
###                        name of a file containing words (one or more
###                        per line) to be added to the internal list of
###                        words that are ignored when forming title
###                        abbreviations, and corporate name
###                        abbreviations.  Multiple --ignore-file options
###                        may be specified.
###
###                        The --long-corporate-names option requests
###                        that corporate names not be abbreviated.
###                        Without this option, an author/editor string
###                        of "{Free Software Foundation}" is reduced to
###                        "FSF".  With this option, it becomes
###                        "Free-Software-Foundation".  This option may
###                        be abbreviated --lo.
###
###                        The --used-file option is followed by the
###                        name of a file containing citation labels
###                        (one per line) to be added to the internal
###                        list of labels known to be already in use.
###                        Multiple --used-file options may be
###                        specified.
###
###                        The --version option produces a version
###                        number report, and then execution immediately
###                        terminates with a success return code (0).
###
###                        The checksum field above contains a CRC-16
###                        checksum as the first value, followed by the
###                        equivalent of the standard UNIX wc (word
###                        count) utility output of lines, words, and
###                        characters.  This is produced by Robert
###                        Solovay's checksum utility.",
###  }
### ====================================================================

### Set environment variables that control sort behavior to the same
### values on all systems, to ensure one consistent sort order (ASCII)
### everywhere.
LC_ALL=C 
LC_COLLATE=C 
LC_CTYPE=C
LC_MESSAGES=C 
LANG=C 

export LC_ALL 
export LC_COLLATE 
export LC_CTYPE
export LC_MESSAGES 
export LANG 

### These four variables are defined by ./configure when it converts
### this file to biblabel.sh:
AWK=/usr/local/bin/mawk
BIBCLEAN=/usr/local/bin/bibclean
VERSIONNUMBER=0.04
VERSIONDATE=10-Mar-2001

### Do NOT change this assignment: it gets modified by "make install"
BIBLABELAWK=`dirname $0`/biblabel.awk

### Establish initial defaults, possibly to be overridden later by
### command-line options. All file variables are whitespace-separated
### lists of files.
CORPORATEFILES=
DUMPCORPORATEFILES=
DUMPIGNOREFILES=
DUMPLABELFILES=
IGNOREFILES=
INPUTFILES=
LONGCORPORATENAMES=0
USEDFILES=

### ====================================================================
### Prevent security attacks from redefinition of IFS!  The definition
### here is a 3-character string: SPACE TAB NEWLINE
IFS=" "
IFS="	$IFS"
IFS="
$IFS"
###??? echo DEBUG: "$IFS" | od -xba ; exit 0

PATH=/bin:/usr/bin:/usr/ucb:$PATH
export PATH

### Shell functions to shorten and clarify option-loop coding below

### --------------------------------------------------------------------

author()
{
	## Print an author credit on stderr.
	##
	## Usage: author

	message 'Author:'
	message '       Nelson H. F. Beebe'
	message '       Center for Scientific Computing'
	message '       University of Utah'
	message '       Department of Mathematics, 322 INSCC'
	message '       155 S 1400 E RM 233'
	message '       Salt Lake City, UT 84112-0090'
	message '       USA'
	message '       Email: beebe@math.utah.edu, beebe@acm.org,'
	message '              beebe@computer.org, beebe@ieee.org (Internet)'
	message '       WWW URL: http://www.math.utah.edu/~beebe'
	message '       Telephone: +1 801 581 5254'
	message '       FAX: +1 801 585 1640, +1 801 581 4148'
	exit 0

}

### --------------------------------------------------------------------

check_for_new_file()
{
	## Usage: check_for_new_file filename

	test -z "$1" && error "check_for_new_file: expected new filename"
	test -f $1 && error "output file $1 already exists"
}

### --------------------------------------------------------------------

check_for_old_file()
{
	## Usage: check_for_old_file filename

	test -z "$1" && error "check_for_old_file: expected existing filename"
	test -f $1 -a -r $1 || error "input file $1 does not exist, or is not readable"
}

### --------------------------------------------------------------------

copyright()
{
	## Print a copyright message on stderr.
	##
	## Usage: copyright

	message '########################################################################'
	message '########################################################################'
	message '########################################################################'
	message '###                                                                  ###'
	message '###      biblabel: generate standardized BibTeX citation labels      ###'
	message '###                                                                  ###'
	message '###     Copyright (C) 1994, 1996, 1997, 2001 Nelson H. F. Beebe      ###'
	message '###                                                                  ###'
	message '### This program is covered by the GNU General Public License (GPL), ###'
	message '### version 2 or later, available as the file COPYING in the program ###'
	message '### source distribution, and on the Internet at                      ###'
	message '###                                                                  ###'
	message '###               ftp://ftp.gnu.org/gnu/GPL                          ###'
	message '###                                                                  ###'
	message '###               http://www.gnu.org/copyleft/gpl.html               ###'
	message '###                                                                  ###'
	message '### This program is free software; you can redistribute it and/or    ###'
	message '### modify it under the terms of the GNU General Public License as   ###'
	message '### published by the Free Software Foundation; either version 2 of   ###'
	message '### the License, or (at your option) any later version.              ###'
	message '###                                                                  ###'
	message '### This program is distributed in the hope that it will be useful,  ###'
	message '### but WITHOUT ANY WARRANTY; without even the implied warranty of   ###'
	message '### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    ###'
	message '### GNU General Public License for more details.                     ###'
	message '###                                                                  ###'
	message '### You should have received a copy of the GNU General Public        ###'
	message '### License along with this program; if not, write to the Free       ###'
	message '### Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   ###'
	message '### MA 02111-1307 USA                                                ###'
	message '########################################################################'
	message '########################################################################'
	message '########################################################################'
	exit 0
}

### --------------------------------------------------------------------

error()
{
	## Print an error message on stderr and exit with a failure
	## status code.
	##
	## Usage: error message

	message "Error: $1"
	exit 1
}

### --------------------------------------------------------------------

message()
{
	## Print a message on stderr. All output from this script must go
	## through this function!

	echo "$1" 1>&2
}

### --------------------------------------------------------------------

usage()
{
	## Issue a usage message and exit with specified code.
	##
	## Usage: usage exit-code [optional-initial-message]

	test -n "$2" && message "Error: $2"
	message 'Usage: '$0' [ --? ]'
	message '                [ --author ]'
	message '                [ --copyright ]'
	message '                [ --corporate-file input-corporate-ignore-file ]'
	message '                [ --dump-corporate-file output-file ]'
	message '                [ --dump-ignore-file output-file ]'
	message '                [ --dump-label-file output-file ]'
	message '                [ --help ]'
	message '                [ --ignore-file input-ignore-file ]'
	message '                [ --long-corporate-names ]'
	message '                [ --used-file input-labels-in-use-file ]'
	message '                [ --version ]'
	message '                [ BibTeXfile(s) ]  >output-citesub-file'
	exit $1
}

### --------------------------------------------------------------------

version()
{
	message "This is $0 version $VERSIONNUMBER [$VERSIONDATE]"
	exit 0
}

### ====================================================================
### Loop over the command-line arguments, collecting biblabel options
### and file names.
while [ $# -gt -0 ]
do
	case `echo $1 | tr A-Z a-z` in

	--'?' | -'?' )
		usage 0
		;;


	--author | --autho | --auth | --aut | --au | --a | \
	-author | -autho | -auth | -aut | -au | -a )
		author
		;;


	--copyleft |  --copylef |  --copyle |  --copyl | \
	-copyleft |  -copylef |  -copyle |  -copyl | \
	--copyright |  --copyrigh |  --copyrig |  --copyri |  --copyr |  --copy |  --cop |  --co | --c | \
	-copyright |  -copyrigh |  -copyrig |  -copyri |  -copyr |  -copy |  -cop |  -co | -c )
		copyright
		;;


	--corporate-file | --corporate-fil | --corporate-fi | --corporate-f | \
	--corporate- | --corporate | --corporat | --corpora | --corpor | \
	--corpo | --corp | --cor | \
	-corporate-file | -corporate-fil | -corporate-fi | -corporate-f | \
	-corporate- | -corporate | -corporat | -corpora | -corpor | \
	-corpo | -corp | -cor )
		shift
		check_for_old_file $1
		CORPORATEFILES="$CORPORATEFILES $1"
		;;


	--dump-corporate-file | --dump-corporate-fil | --dump-corporate-fi | \
	--dump-corporate-f | --dump-corporate- | --dump-corporate | \
	--dump-corporat | --dump-corpora | --dump-corpor | --dump-corpo | \
	--dump-corp | --dump-cor | --dump-co | --dump-c | \
	-dump-corporate-file | -dump-corporate-fil | -dump-corporate-fi | \
	-dump-corporate-f | -dump-corporate- | -dump-corporate | \
	-dump-corporat | -dump-corpora | -dump-corpor | -dump-corpo | \
	-dump-corp | -dump-cor | -dump-co | -dump-c )
		shift
		check_for_new_file $1
		DUMPCORPORATEFILES="$DUMPCORPORATEFILES $1"
		;;


	--dump-ignore-file | --dump-ignore-fil | --dump-ignore-fi | \
	--dump-ignore-f | --dump-ignore- | --dump-ignore | --dump-ignor | \
	--dump-igno | --dump-ign | --dump-ig | --dump-i | \
	-dump-ignore-file | -dump-ignore-fil | -dump-ignore-fi | \
	-dump-ignore-f | -dump-ignore- | -dump-ignore | -dump-ignor | \
	-dump-igno | -dump-ign | -dump-ig | -dump-i )
		shift
		check_for_new_file $1
		DUMPIGNOREFILES="$DUMPIGNOREFILES $1"
		;;


	--dump-label-file | --dump-label-fil | --dump-label-fi | \
	--dump-label-f | --dump-label- | --dump-label | --dump-labe | \
	--dump-lab | --dump-la | --dump-l | \
	-dump-label-file | -dump-label-fil | -dump-label-fi | \
	-dump-label-f | -dump-label- | -dump-label | -dump-labe | \
	-dump-lab | -dump-la | -dump-l )
		shift
		check_for_new_file $1
		DUMPLABELFILES="$DUMPLABELFILES $1"
		;;


	--help | --hel | --he | --h | -help | -hel | -he | -h )
		usage 0
		;;


	--ignore-file | --ignore-fil | --ignore-fi | --ignore-f | --ignore- | \
	--ignore | --ignor | --igno | --ign | --ig | --i | \
	-ignore-file | -ignore-fil | -ignore-fi | -ignore-f | -ignore- | \
	-ignore | -ignor | -igno | -ign | -ig | -i )
		shift
		check_for_old_file $1
		IGNOREFILES="$IGNOREFILES $1"
		;;


	--long-corporate-names | --long-corporate-name | --long-corporate-nam | --long-corporate-na | \
	--long-corporate-n | --long-corporate- | --long-corporate | --long-corporat | \
	--long-corpora | --long-corpor | --long-corpo | --long-corp | --long-cor | --long-co | \
	--long-c | --long- | --long | --lon | --lo | \
	-long-corporate-names | -long-corporate-name | -long-corporate-nam | -long-corporate-na | \
	-long-corporate-n | -long-corporate- | -long-corporate | -long-corporat | \
	-long-corpora | -long-corpor | -long-corpo | -long-corp | -long-cor | -long-co | \
	-long-c | -long- | -long | -lon | -lo )
		LONGCORPORATENAMES=1
		;;



	--version | --versio | --versi | --vers | --ver | --ve | --v | \
	-version | -versio | -versi | -vers | -ver | -ve | -v )
		version
		;;


	--f | -f | \
	--used-file | --used-fil | --used-fi | --used-f | --used- | --used | \
	--use | --us | --u | \
	-used-file | -used-fil | -used-fi | -used-f | -used- | -used | \
	-use | -us | -u )
		shift
		check_for_old_file $1
		USEDFILES="$USEDFILES $1"
		;;


	-*)
		usage 1 "unrecognized option: [$1]"
		;;


	*)
		check_for_old_file $1
		INPUTFILES="$INPUTFILES $1"
		;;

	esac
	shift			# discard this option
done

$BIBCLEAN -no-check -no-warning -max-width 0 $INPUTFILES | \
	$AWK -f $BIBLABELAWK \
		-v CORPORATEFILES="$CORPORATEFILES" \
		-v DUMPCORPORATEFILES="$DUMPCORPORATEFILES" \
		-v DUMPIGNOREFILES="$DUMPIGNOREFILES" \
		-v DUMPLABELFILES="$DUMPLABELFILES" \
		-v IGNOREFILES="$IGNOREFILES" \
		-v INPUTFILES="$INPUTFILES" \
		-v LONGCORPORATENAMES=$LONGCORPORATENAMES \
		-v USEDFILES="$USEDFILES" \
		-v VERSIONDATE="$VERSIONDATE" \
		-v VERSIONNUMBER="$VERSIONNUMBER"
