/* -*-C-*- xalloc.c */
/*-->xalloc*/
/**********************************************************************/
/****************************** xalloc ********************************/
/**********************************************************************/

/***********************************************************************

In most applications that obtain dynamic memory via calls to the
standard library routines, calloc(), malloc(), or realloc(), failure
necessitates job termination.  xcalloc(), xmalloc(), and xrealloc()
therefore provide convenient interfaces to the standard routines which
terminate on failure; their callers need not check the returned value.

This file provides private versions xcalloc(), xmalloc(), xrealloc(),
and xfree() that terminate with an error message if the request cannot
be satisfied.  xmemused() can be called to find out how much memory
has been allocated via calls to the others; the value returned does not
include storage allocated for internal housekeeping by these routines.

Some implementations of the standard library routines abort, or return
an error, if a non-positive number of bytes is requested.  Some
library realloc()'s fail if their first argument is NULL.  Both these
defects are remedied by the versions here.

A simple test program is present and can be compiled if TEST is
defined at compile time.

The implementations here add an extra word pair before the end of each
allocated block.  The first is a magic number used to check for
corruption, or passing of a block not allocated by these routines, and
the second is the number of bytes found.  A block allocated by the
standard routines can be safely passed to these routines, but the
private space accounting will not consider it.  The converse is NOT
true, however; a block allocated by x*alloc() cannot be successfully
freed by free(), only by xfree().

Although size_t is the (unsigned) type of quantities passed to the
standard library allocation routines, on PC DOS at least, it may not
be large enough to hold the total number of bytes allocated, so we
intentionally use unsigned long for that quantity.

These routines were inspired by routines of the same names in GNU
Emacs, and by code posted to the INFO-C bulletin board by Karl Heuer
(karl@haddock.isc.com) in September 1988.  Numerous Changes have been
incorporated to enhance portability.

[26-Jan-1989]
***********************************************************************/

#include <config.h>

#if defined(HAVE_STDIO_H)
#include <stdio.h>
#endif

#if defined(HAVE_STDLIB_H)
#include <stdlib.h>
#endif

#include <xalloc.h>

RCSID("$Id: xalloc.c,v 1.11 1994/03/01 02:35:35 beebe Exp $")


#if    IBM_PC_TURBO
#include <alloc.h>			/* for coreleft() declaration */
#endif /* IBM_PC_TURBO */

#ifdef DVI
#include <typedefs.h>
extern UNSIGN32 debug_code;		/* 0 for no debug output */
#define  DBGOPT(flag)	(debug_code & (flag))
#define  DBG_MALLOC	0x0100
#endif /* DVI */

/***********************************************************************
The memory block manipulated here is
			*********
bytes in DATA block->	* SIZE  * <- pointer returned by *alloc()
			*********
Magic cookie->		* MAGIC *
			*********
			*  pad	* <- boundary padding (usually absent)
			*********
User data->		* DATA  * <- pointer returned by x*alloc()
			*       *
			*       *
			*********

ALIGN is the byte address boundary alignment value; it may be
system-dependent.  However, I'm not aware of any system for which it is
larger than 8.
************************************************************************/

#ifdef DVI
/* ADR() is used to format an address for printing in the native form.
Since the result is returned in static storage, ADR() must not be
invoked more than once in an argument list or expression. */

static char		adrtmp[20];
#if __STDC__
#define ADR(x)		(sprintf(adrtmp, "%p", (void*)(x)), adrtmp)
#else /* NOT __STDC__ */
#define ADR(x)		(sprintf(adrtmp, "16#%lx", (unsigned long)(x)), adrtmp)
#endif /* __STDC__ */
#endif /* DVI */

#define ALIGN   	((size_t)8)
#define MAGIC_NUMBER	((size_t)(0x12345678L))
#define HEADER_SIZE	(size_t)((((2*sizeof(size_t)+ALIGN-1)/ALIGN)*ALIGN))
#define DATA(p)		(VOIDP)((char*)p + HEADER_SIZE)
#define HEADER(p)	(VOIDP)((char*)p - HEADER_SIZE)

#define MAGIC(p)	*((size_t*)(p) + 1)
#define SIZE(p)		*((size_t*)p)

#define ULONG		unsigned long		/* convenient shorthand */

static void		xabort ARGS((const char *name_));
static ULONG		bytes_allocated = (ULONG)0;

#ifdef DVI

static void		dbgmem ARGS((const char *caller, VOIDP block,
			    size_t size, size_t magic,ULONG usage));

#if sun
			/* missing from Sun C and G++ headers */
			/* but present in Sun CC (C++) headers */
#if !defined(__cplusplus)
int			malloc_debug ARGS((int));
#endif /* !defined(__cplusplus) */

#if (defined(__cplusplus) && __GNUC__)
extern "C"
{
int			malloc_debug ARGS((int));
};
#endif /* (defined(__cplusplus) && __GNUC__) */

#endif /* sun */

static void
#if STDC
dbgmem(
const char	*caller,
VOIDP		block,
size_t		size,
size_t		magic,
ULONG		usage
)
#else /* NOT STDC */
dbgmem(caller,block,size,magic,usage)
const char	*caller;
VOIDP		block;
size_t		size;
size_t		magic;
ULONG		usage;
#endif /* STDC */
{
    (void)fprintf(stderr,"%s():\tBlock = %s\tsize = %5ld\
\tmagic = 16#%lx\tusage = %5lu\n",
	caller, ADR(block), (long)size, (unsigned long)magic, usage);
}
#endif /* DVI */

#ifdef TEST
#include <stdio.h>

int			main ARGS((void));

int
main(VOID_ARG)			/* simple test -- silence means success */
{
    VOIDP p1;
    p1 = xmalloc(10);
    (void)xmalloc(99);
    xfree(p1);
    if (xmemused() != (ULONG)99)
    {
	(void)printf(
	    "Error: should be 99 bytes allocated; xmemused() says %lu.\n",
	    xmemused());
	exit(EXIT_FAILURE);
    }
    else
	exit(EXIT_SUCCESS);
    return (0);
}
#endif /* TEST */


static
void
#if STDC
xabort(
const char  *name
)
#else /* NOT STDC */
xabort(name)
const char  *name;
#endif /* STDC */
{
    char message[80];

#if    IBM_PC_TURBO
    /* Use Turbo C function to tell how much memory is left. */
    (void)sprintf(message,
    "%s()--out of memory.  %lu bytes are allocated and %lu bytes are free\n",
	name,bytes_allocated,(ULONG)coreleft());
#else  /* NOT IBM_PC_TURBO */
    (void)sprintf(message,"%s()--out of memory.  %lu bytes are allocated\n",
	name,bytes_allocated);
#endif /* IBM_PC_TURBO */

    (void)perror(message);
    exit(EXIT_FAILURE);
}


VOIDP
#if STDC
xcalloc(
size_t nmemb,
size_t size
)
#else /* NOT STDC */
xcalloc(nmemb, size)		/* allocate nelem*elsize bytes, or die */
size_t nmemb;
size_t size;
#endif /* STDC */
{
    register VOIDP p;

    size *= nmemb;
    p = (VOIDP)calloc((size_t)(HEADER_SIZE + size),1);
    if (p == (VOIDP)NULL)
	xabort("xmalloc");
    SIZE(p) = size;
    MAGIC(p) = MAGIC_NUMBER;
    bytes_allocated += (ULONG)size;

#ifdef DVI
    if (DBGOPT(DBG_MALLOC))
	dbgmem("xcalloc", DATA(p), SIZE(p), MAGIC(p), bytes_allocated);
#endif /* DVI */

    return (DATA(p));
}


void
#if STDC
xfree(
VOIDP p
)
#else /* NOT STDC */
xfree(p)
VOIDP p;
#endif /* STDC */
{
    VOIDP q;

    if (p != (VOIDP)NULL)
    {
	q = HEADER(p);
	if (MAGIC(q) == MAGIC_NUMBER)
	{
	    bytes_allocated -= SIZE(q);
	    p = q;
	}

#ifdef DVI
    if (DBGOPT(DBG_MALLOC))
	dbgmem("xfree", DATA(p), SIZE(p), MAGIC(p), bytes_allocated);
#endif /* DVI */

	free((VOIDP)p);
    }
}


VOIDP
#if STDC
xmalloc(
size_t size
)
#else /* NOT STDC */
xmalloc(size)			/* allocate size bytes, or die */
size_t size;
#endif /* STDC */
{
    register VOIDP p;

#ifdef DVI
#if sun
#if !__solaris
    if (DBGOPT(DBG_MALLOC))
	malloc_debug(2);	/* turn on full heap examination */
#endif /* __solaris */
#endif /* sun */
#endif /* DVI */

    p = (VOIDP)malloc(HEADER_SIZE + size);
    if (p == (VOIDP)NULL)
	xabort("xmalloc");
    SIZE(p) = size;
    MAGIC(p) = MAGIC_NUMBER;
    bytes_allocated += (ULONG)size;

#ifdef DVI
    if (DBGOPT(DBG_MALLOC))
	dbgmem("xmalloc", DATA(p), SIZE(p), MAGIC(p), bytes_allocated);
#endif /* DVI */

    return (DATA(p));
}


ULONG
xmemused(VOID_ARG)
{
    return (bytes_allocated);
}


VOIDP
#if STDC
xrealloc(
VOIDP block,			/* existing block (may be NULL) */
size_t size			/* new size */
)
#else /* NOT STDC */
xrealloc(block,size)		/* reallocate block with size bytes, or die */
VOIDP block;			/* existing block (may be NULL) */
size_t size;			/* new size */
#endif /* STDC */
{
    register VOIDP p;
    register VOIDP q;

    if (block == (VOIDP)NULL)	/* new allocation */
	p = (VOIDP)malloc(HEADER_SIZE + size);
    else			/* reallocation of old block */
    {
	p = block;
	q = HEADER(p);
	if (MAGIC(q) == MAGIC_NUMBER)
	{
	    bytes_allocated -= SIZE(q);
	    p = q;
	}
	p = (VOIDP)realloc(p,HEADER_SIZE + size);
	if (p == (VOIDP)NULL)
	    xabort("xrealloc");
    }
    if (p == (VOIDP)NULL)
	xabort("xrealloc");
    bytes_allocated += size;
    SIZE(p) = size;
    MAGIC(p) = MAGIC_NUMBER;

#ifdef DVI
    if (DBGOPT(DBG_MALLOC))
	dbgmem("xrealloc", DATA(p), SIZE(p), MAGIC(p), bytes_allocated);
#endif /* DVI */

    return (DATA(p));
}
