#!/bin/sh
### ====================================================================
###  @UNIX-shell-file{
###     author          = "Nelson H. F. Beebe",
###     version         = "0.10",
###     date            = "04 December 1998",
###     time            = "11:00:59 MST",
###     filename        = "biborder.sh",
###     address         = "Center for Scientific Computing
###                        University of Utah
###                        Department of Mathematics, 322 INSCC
###                        155 S 1400 E RM 233
###                        Salt Lake City, UT 84112-0090
###                        USA",
###     telephone       = "+1 801 581 5254",
###     FAX             = "+1 801 585 1640, +1 801 581 4148",
###     checksum        = "56789 870 3019 25412",
###     email           = "beebe@math.utah.edu, beebe@acm.org,
###                        beebe@ieee.org (Internet)",
###     codetable       = "ISO/ASCII",
###     keywords        = "bibliography, ordering, BibTeX",
###     supported       = "yes",
###     docstring       = "This file contains the biborder utility, a
###                        program for arranging key/value pairs into
###                        a standard order in BibTeX data base files.
###
###                        Indentation is preserved on all lines, but
###                        trailing whitespace is discarded.  Entries
###                        themselves, and any comments present
###                        between entries, are NOT reordered.  The
###                        companion utility, bibsort(1), provides
###                        that service.
###
###                        Syntax errors in the input stream will
###                        cause abrupt termination with a fatal error
###                        message and a non-zero exit code.  The
###                        output will be incomplete, so you should
###                        always verify file line counts (input and
###                        output counts should be identical) before
###                        assuming that you can replace the input
###                        file with the output file.  If nawk is
###                        used, biborder will verify input and output
###                        line counts when it terminates normally at
###                        end of file; it cannot do so with gawk.
###
###                        Usage:
###                              biborder
###                                     [-alphabetical]
###					[-bimonthly m1/m2/m3/m4/m5/m6 ]
###					[-check-missing]
###					[-monthly]
###					[-quarterly m1/m2/m3/m4 ]
###					bibfile(s) >outfile
###                        or
###                              biborder
###                                     [-alphabetical]
###					[-bimonthly m1/m2/m3/m4/m5/m6 ]
###					[-check-missing]
###					[-monthly]
###					[-quarterly m1/m2/m3/m4 ]
###					<infile >outfile
###
###			   Switch names may be abbreviated to the
###			   minimal unique prefix.
###
###                        If -alphabetical is specified, the standard
###                        order of keys is ignored, and strict
###                        alphabetical order is used instead.
###
###			   If -check-missing is specified, missing
###			   expected fields will be supplied, with the
###			   field name prefixed with OPT, and the value
###			   string set to a pair of question marks,
###			   e.g.  OPTvolume = "??".  The OPT prefix
###			   ensures that the key is ignored by BibTeX
###			   (and thus that the question marks will not
###			   appear in an output .bbl file), and
###			   together with the question marks,
###			   highlights the missing data.  In addition,
###			   the GNU Emacs bibtex-mode editing support
###			   has convenient functions for removing the
###			   OPT prefixes, and so does bibclean(1).
###
###			   If -bimonthly is specified, all of the
###			   journals in the BibTeX file entries are
###			   assumed to appear at two-month
###			   intervals. The six month names that
###			   correspond to each such interval must be
###			   given as a SINGLE argument following
###			   -bimonthly, with the names separated by
###			   (any number of) non-letters.  A typical
###			   example is
###
###			   feb-apr-jun-aug-oct-dec
###
###			   specifying that issue number 1 appears in
###			   February, and number 6 in December.
###
###			   If -monthly is specified, all of the
###			   journals in the BibTeX file entries are
###			   assumed to appear monthly, 1 in January, 2
###			   in February, ..., 12 in December.  If only
###			   one of the number and month values is
###			   supplied, biborder will supply the other
###			   automatically, but will spell the key name
###			   in uppercase letters to distinguish it (a
###			   simple editor string substitution, or
###			   bibclean(1), can subsequently normalize the
###			   appearance).  If both month and number are
###			   specified in a BibTeX entry, they are
###			   checked for consistency, and a warning is
###			   issued if they do not match.
###
###			   If -quarterly is specified, all of the
###			   journals in the BibTeX file entries are
###			   assumed to appear quarterly. The four month
###			   names that correspond to each quarter must
###			   be given as a SINGLE argument following
###			   -quarterly, with the names separated by
###			   (any number of) non-letters.  Some examples
###			   are
###
###				   jan-apr-jul-oct
###				   mar////jun.sep:dec
###				   Spring\ Summer\ Fall\ Winter
###				   'Spring Summer Fall Winter'
###
###			   The last two examples show that when space
###			   separators are used, they must be
###			   backslashed, or else the list must be
###			   surrounded by quotes, to force UNIX shells
###			   to interpret the quarter names as a single
###			   argument.
###
###			   The -bimonthly, -monthly, and -quarterly
###			   options are mutually exclusive, and should
###			   only be used in journal-specific
###			   bibliographies, since number-month pairing
###			   does not hold for all journals.
###
###                        All remaining command-line words are assumed
###                        to be input files.  Should such a filename
###                        begin with a hyphen, it must be disguised by
###                        a leading absolute or relative directory
###                        path, e.g. /tmp/-foo.bib or ./-foo.bib.
###
###                        WARNINGS:
###
###                        (1) This simple version does NOT recognize
###                        bib entries with outer parentheses instead of
###                        braces, or with line breaks between the @Name
###                        and following opening brace.  Use bibclean(1)
###                        to standardize and syntax check the
###                        bibliography entries first.
###
###                        (2) Implementation limitations in nawk or
###                        gawk may result in premature termination
###                        because of maximum string lengths being
###                        exceeded.  This can happen with long
###                        abstract or summary strings.  This problem
###                        has been seen more frequently with some
###                        UNIX nawk implementations than with the
###                        Free Software Foundation GNU Project's
###                        gawk, so we prefer to use gawk, if
###                        available.
###
###                        The checksum field above contains a CRC-16
###                        checksum as the first value, followed by the
###                        equivalent of the standard UNIX wc (word
###                        count) utility output of lines, words, and
###                        characters.  This is produced by Robert
###                        Solovay's checksum utility.",
###  }
########################################################################

# Assign default initial values
ALPHABETICAL=0
FILES=
BIMONTHLY=0
CHECKMISSING=0
MONTHLY=0
QUARTERLY=0

# Loop over the command-line arguments, collecting biborder switches,
# and file names.
while [ $# -gt -0 ]
do
	case $1 in
	-a*)
		ALPHABETICAL=1
		;;
	-b*)
		BIMONTHLY=1
		shift
		MONTHNAMES="$1"
		;;
	-c*)
		CHECKMISSING=1
		;;
	-m*)
		MONTHLY=1
		;;
	-q*)
		QUARTERLY=1
		shift
		MONTHNAMES="$1"
		;;
	-*)
		echo Usage: $0
		echo '		[-bimonthly m1/m2/m3/m4/m5/m6]'
		echo '		[-check-missing]'
		echo '		[-monthly]'
		echo '		[-quarterly m1/m2/m3/m4]'
		echo '		BibTeXfiles or <infile'
		echo '		>outfile'
		exit 1
		;;
	*)			# everything else is assumed to be a filename
		FILES="$FILES $1"
		;;
	esac
	shift			# discard this switch or filename
done

# We store the awk program as a (large) string constant
PROGRAM='
BEGIN {
	stderr = "/dev/tty"	# nawk
	stderr = "/dev/stderr"	# gawk
	BIMONTHLY += 0		# coerce string to number
	CHECKMISSING += 0
	MONTHLY += 0
	QUARTERLY += 0
	UNKNOWN_VALUE = "??"
	# print "DEBUG: MONTHLY=" MONTHLY " BIMONTHLY=" BIMONTHLY " \
	#	QUARTERLY=" QUARTERLY " MONTHNAMES=" MONTHNAMES
	if (((QUARTERLY != 0) + (MONTHLY != 0) + (BIMONTHLY != 0)) > 1)
	    error("Only ONE of -bimonthly, -monthly, " \
		"and -quarterly may be specified")
	initialize_months()
}

/^[ \t]*@[ \t]*[Pp][Rr][Ee][Aa][Mm][Bb][Ll][Ee][ \t]*{/ {
	trim()
	squeeze()
	k = index($0,"{") + 1
	print_braced_item()
	next
}

/^[ \t]*@[ \t]*[sS][tT][rR][iI][nN][gG][ \t]*{/ {
	trim()
	squeeze()
	print_braced_item()
	next
}

# "@keyword{label,"
/^[ \t]*@[ \t]*[a-zA-Z0-9]*[ \t]*{/       {
	item = collect_braced_item()
	print_item(item)
	next
}

{				# all other line types match this
	trim()
	print
	next
}

END {
# gawk does nt implement ONR, sigh...
#    if (ONR != NR)
#	error("Input and output line counts differ: input=" NR \
#	    ", output=" ONR)
}

function add_key_abbrev_pair(key,abbrev)
{
    key_value_pair[key] = "  OPT" key " =" \
	substr("                 ",1,17 - (5 + length(key) + 2)) abbrev ","
}

function add_key_value_pair(key,value)
{
    key_value_pair[key] = "  OPT" key " =" \
	substr("                 ",1,17 - (5 + length(key) + 2)) \
	"\"" value "\","
}

function brace_count(s, k,n)
{
    n = 0
    for (k = 1; k <= length(s); ++k)
    {
	if (substr(s,k,1) == "{")
	    n++
	else if (substr(s,k,1) == "}")
	    n--
    }
    return (n)
}

function check_article()
{
    check_missing_key("author")
    check_missing_key("title")
    check_missing_key("journal")
    check_missing_key("volume")
    check_missing_key("number")
    check_missing_key("pages")
    check_missing_key("month")
    check_missing_key("year")
}

function check_author_editor()
{
    if (!("author" in key_value_pair) && !("editor" in key_value_pair))
	add_key_value_pair("author",UNKNOWN_VALUE)
}

function check_book()
{
    check_author_editor()
    check_missing_key("title")
    check_missing_key("publisher")
    check_missing_key("address")
    check_missing_ISBN()
    check_missing_key("LCCN")
    check_missing_key("pages")
    check_missing_key("year")
}

function check_booklet()
{
    check_missing_key("author")
    check_missing_key("title")
    check_missing_key("howpublished")
    check_missing_key("address")
    check_missing_key("year")
}

function check_inbook()
{
    check_author_editor()
    check_missing_key("title")
    check_missing_key("chapter")
    if (!("crossref" in key_value_pair))
    {
	check_missing_key("publisher")
	check_missing_key("address")
	check_missing_ISBN()
	check_missing_key("LCCN")
    }
    check_missing_key("pages")
    check_missing_key("year")
}

function check_incollection()
{
    check_author_editor()
    check_missing_key("title")
    if (!("crossref" in key_value_pair))
    {
	check_missing_key("booktitle")
	check_missing_key("publisher")
	check_missing_key("address")
	check_missing_ISBN()
	check_missing_key("LCCN")
    }
    check_missing_key("pages")
    check_missing_key("year")
}

function check_inproceedings()
{
    check_author_editor()
    check_missing_key("title")
    if (!("crossref" in key_value_pair))
    {
	check_missing_key("booktitle")
	check_missing_key("publisher")
	check_missing_key("address")
	check_missing_ISBN()
	check_missing_key("LCCN")
    }
    check_missing_key("pages")
    check_missing_key("year")
}

function check_manual()
{
    check_author_editor()
    check_missing_key("title")
    check_missing_key("organization")
    check_missing_key("address")
    check_missing_key("pages")
    check_missing_key("year")
}

function check_mastersthesis()
{
    check_missing_key("author")
    check_missing_key("title")
    check_missing_key("school")
    check_missing_key("address")
    check_missing_key("type")
    check_missing_key("month")
    check_missing_key("year")
}

function check_misc()
{
    check_missing_key("author")
    check_missing_key("title")
    check_missing_key("howpublished")
    check_missing_key("year")
}

function check_missing()
{
    if      (entry_type == "article")		check_article()
    else if (entry_type == "book")		check_book()
    else if (entry_type == "booklet")		check_booklet()
    else if (entry_type == "inbook")		check_inbook()
    else if (entry_type == "incollection")	check_incollection()
    else if (entry_type == "inproceedings")	check_inproceedings()
    else if (entry_type == "manual")		check_manual()
    else if (entry_type == "mastersthesis")	check_mastersthesis()
    else if (entry_type == "misc")		check_misc()
    else if (entry_type == "periodical")	check_periodical()
    else if (entry_type == "phdthesis")		check_phdthesis()
    else if (entry_type == "proceedings")	check_proceedings()
    else if (entry_type == "techreport")	check_techreport()
    else if (entry_type == "unpublished")	check_unpublished()
    else
	warning("unrecognized entry type [" entry_type "]")
}

function check_missing_ISBN()
{
    if (!("ISBN" in key_value_pair) && \
	("year" in key_value_pair) && \
	((0 + get_value("year",key_value_pair["year"])) > 1971))
	add_key_value_pair("ISBN",UNKNOWN_VALUE)
}

function check_missing_key(key)
{
    if (!(key in key_value_pair))
	add_key_value_pair(key,UNKNOWN_VALUE)
}

function check_number_month( month_value,number_value,quote)
{
    if ("month" in key_value_pair)
    {
	month_value = tolower(get_value("month",key_value_pair["month"]))
	if ("number" in key_value_pair)
	{
	    number_value = get_value("number",key_value_pair["number"])
	    if (number[month_value] != number_value)
		warning("number " number_value \
		    " does not match month " month_value)
	}
	else if (month_value in number)
	    add_key_value_pair("number",number[month_value])
	else
	{
	    add_key_value_pair("number",UNKNOWN_VALUE)
	    warning("unrecognized month [" month_value "]")
	}
    }
    else if ("number" in key_value_pair)
    {
	number_value = get_value("number",key_value_pair["number"])
	if ("month" in key_value_pair)
	{
	    month_value = to_lower(get_value("month",key_value_pair["month"]))
	    if (number[month_value] != number_value)
		warning("number " month_value \
		    " does not match month " month_value)
	}
	else if (number_value in month)
	{
	    quote = (12 in month) ? "" : "\""
	    add_key_abbrev_pair("month", month[number_value])
	}
	else
	{
	    add_key_value_pair("month",UNKNOWN_VALUE)
	    warning("number [" number_value "] not in range " \
		(MONTHLY ? "1..12" \
			 : (BIMONTHLY ? "1..6" \
				      : (QUARTERLY ? "1..4" \
						   : UNKNOWN_VALUE))))
	}
    }
}

function check_periodical()
{
    check_missing_key("key")
    check_missing_key("address")
    check_missing_key("ISSN")
    check_missing_key("LCCN")
    check_missing_key("publisher")
    check_missing_key("title")
}

function check_phdthesis()
{
    check_missing_key("author")
    check_missing_key("title")
    check_missing_key("school")
    check_missing_key("address")
    check_missing_key("type")
    check_missing_key("month")
    check_missing_key("year")
}

function check_proceedings()
{
    if (!("author" in key_value_pair) && \
	!("editor" in key_value_pair))
	check_missing_key("key")
    check_missing_key("title")
    check_missing_key("publisher")
    check_missing_key("address")
    check_missing_ISBN()
    check_missing_key("LCCN")
    check_missing_key("pages")
    check_missing_key("year")
}

function check_techreport()
{
    check_missing_key("author")
    check_missing_key("title")
    check_missing_key("institution")
    check_missing_key("address")
    check_missing_key("type")
    check_missing_key("month")
    check_missing_key("year")
}

function check_unpublished()
{
    check_missing_key("author")
    check_missing_key("title")
    check_missing_key("note")
    check_missing_key("year")
}

function clear_array(array, key)
{
    for (key in array)
	delete array[key]
}

function collect_braced_item( count,item,k,s)
{
    # Starting with the current contents of $0, collect lines until we
    # reach a zero brace count. To guard against infinite loops in the
    # event of unbalanced braces, we abruptly terminate processing if
    # an at-sign is detected in column 1.  This function is used for
    # those entry types that require key/value pair reordering.

    start_fnr = FNR
    start_line = $0
    entry_type = substr($0,2)
    sub(/ *{.*$/,"",entry_type)
    entry_type = tolower(entry_type)
    clear_array(key_value_pair)
    squeeze()
    trim()
    count = brace_count($0)
    item = $0 "\n"
    while (count != 0)
    {
	if (getline <= 0)
	    break
	if (substr($0,1,1) == "@") # should use match($0,/^[ \t]+@/), but
				   # this is faster, and usually correct
	    error("New entry encountered before balanced braces found")
	trim()
	# NB: syntax of abbrev, entry, key, and field names taken from
	# biblex source code: see Nelson H. F. Beebe, "Bibliography
	# prettyprinting and syntax checking", TUGboat 14(3), 222,
	# October (1993) and TUGboat 14(4), 395--419, December (1993).
	# NB: in match() below, \047 is an apostrophe, which we cannot
	# use inside the sh apostrophe-delimited string containing
	# this program.
	# NB: Add underscore, even though it is not in grammar.
	if (match($0,/^[ \t]*[A-Za-z][---A-Za-z0-9_:.+\/\047]*[ \t]*=/))
	    s = collect_key_value_pair()
	else
	    s = $0
	count += brace_count(s)
    }

    if ((BIMONTHLY || MONTHLY || QUARTERLY) && (entry_type == "article"))
	check_number_month()

    if (CHECKMISSING)
	check_missing()

    # After trying strict alphabetic order for the keywords, I decided
    # that it is more useful to have most of them in a specific order,
    # that corresponding to the usual order in an article citation.
    # get_pair() will delete each entry that it extracts, and ignore
    # any that do not exist, so that key/value pairs are not duplicated
    # in the collections after sort_keys().  The list here includes all
    # of the field names defined in the GNU Emacs bibtex.el support
    # code, plus CODEN, day, key, ISBN, ISSN, LCCN, MRnumber, MRclass,
    # MRreviewer, price, and URL.

    if (!ALPHABETICAL)
    {
	item = item get_pair("author")
	item = item get_pair("editor")
	item = item get_pair("key")
	item = item get_pair("booktitle")
	item = item get_pair("title")
	item = item get_pair("crossref")
	item = item get_pair("chapter")
	item = item get_pair("journal")
	item = item get_pair("volume")
	item = item get_pair("type")
	item = item get_pair("number")
	item = item get_pair("howpublished")
	item = item get_pair("institution")
	item = item get_pair("organization")
	item = item get_pair("publisher")
	item = item get_pair("school")
	item = item get_pair("address")
	item = item get_pair("edition")
	item = item get_pair("pages")
	item = item get_pair("day")
	item = item get_pair("month")
	item = item get_pair("year")
	item = item get_pair("CODEN")
	item = item get_pair("ISBN")
	item = item get_pair("ISSN")
	item = item get_pair("LCCN")
	item = item get_pair("MRclass")
	item = item get_pair("MRnumber")
	item = item get_pair("MRreviewer")
	item = item get_pair("bibdate")
	item = item get_pair("bibsource")
	item = item get_pair("note")
	item = item get_pair("price")
	item = item get_pair("series")
	item = item get_pair("URL")
    }
    sort_keys()			# only a few are left at this point

    for (k = 1; sorted_keys[k]; ++k) # output them in ordered by key
	item = item get_pair(sorted_keys[k])

    item = item "}\n"
    return item
}

function collect_key_value_pair( key,s)
{
    # This function is called when a line of the form " key = ..." is met.
    s = $0
    match($0,/[A-Za-z][---A-Za-z0-9_:.+\/\047]*/)
    key = substr($0,RSTART,RLENGTH)
    if (match($0,/^[ \t]*[A-Za-z][---A-Za-z0-9_:.+\/\047]*[ \t]*=[ \t]*\"/))
    {				# then we have key = "...value..."
	# Collect any multiline key/value assignment, using the simple
	# heuristic (guaranteeable by bibclean) that a quoted value string
	# ends with a quote, or quote comma.
	while (match($0,/\",?$/) == 0)
	{
	    if (getline <= 0)
		error("Unexpected end-of-file while collecting key/value pair")
	    if (substr($0,1,1) == "@")	# should use match($0,/^[ \t]+@/), but
					# this is faster, and usually correct
		error("New entry encountered before end of string")
	    trim()
	    s = s "\n" $0
	}
    }
    # else must be key = abbrev, which we assume takes just one line

    # At this point, s is either
    #	key = "string-value",
    # or
    #	key = abbrev-value,
    # (without a trailing newline, but possibly embedded ones)

    if (key in key_value_pair)
    {
	if (key_value_pair[key] ~ "\"[?]*\"")
	{
	    # Previous value is empty, so we silently allow
	    # this new key/value to replace it
	}
	else if (s ~ "\"[?]*\"")
	    s = key_value_pair[key] # replace new empty value by previous non-empty one
	else if (s == key_value_pair[key])
	    ; # identical values, so silent replacement below is okay
	else
	{
	    warning("duplicate key [" key "]")
	    while (key in key_value_pair)	# append -z to get a unique sort key
		key = key "-z"
	}
    }
    key_value_pair[key] = s	# NB: omits final newline
    return (s)
}

function error(msg)
{		# print a message and terminate with failing exit code
    message("??FATAL ERROR:" msg)
    exit(1)
}

function get_pair(key, s)
{
    s = ""
    while (key in key_value_pair)
    {
	# Handle duplicate keys transparently to caller, so that such
	# keys always appear consecutively in the ordered output,
	# independent of the requested sorting order.

	s = s key_value_pair[key] "\n"
	delete key_value_pair[key]
	key = key "-z"
    }
    return s
}

function get_value(key,key_value, s)
{
    if (match(key_value,"^[ \t]*" key "[ \t]*=[ \t]*\""))
    {				# have key = "value"
	s = substr(key_value,RSTART+RLENGTH)
	sub(/\",$/,"",s)
	sub(/\"$/,"",s)
	return s
    }
    else if (match(key_value,"^[ \t]*" key "[ \t]*=[ \t]*[a-zA-Z]"))
    {				# have key = abbrev
	s = substr(key_value,RSTART+RLENGTH-1)
	sub(/,$/,"",s)
    }
    else
	s = ""
    # print "DEBUG: get_value(" key ",[" key_value "]) -> [" s "]"
    return s
}

function initialize_months( k,n,need,option)
{
    if (BIMONTHLY)
    {
	need = 6
	option = "-bimonthly"
    }
    else if (MONTHLY)
    {
	MONTHNAMES = "jan feb mar apr may jun jul aug sep oct nov dec"
	need = 12
	option = "-monthly"
    }
    else if (QUARTERLY)
    {
	need = 4
	option = "-quarterly"
    }
    n = split(MONTHNAMES,month,"[^A-Za-z]")
    if (n != need)
	error(option " [" MONTHNAMES "] needs " need " month names")
    for (k = 1; k <= n; ++k)	# prepare an inverted table
	number[month[k]] = k
}

function message(msg)
{
    print FILENAME ":" FNR ":" msg "\tIn:" start_fnr ":" start_line >stderr
}

function print_braced_item(count)
{
    # Starting with the current contents of $0, print lines until we
    # reach a zero brace count.  This function is used for
    # @Preamble{...} and @String{...}, which require no special
    # processing.

    start_fnr = FNR
    start_line = $0
    count = brace_count($0)
    print $0
    while (count != 0)
    {
	if (getline <= 0)
	    break
	if (substr($0,1,1) == "@")
	    error("New entry encountered before balanced braces found")
	print $0
	count += brace_count($0)
    }
}

function print_item(item)
{
    printf("%s", item)
}

function sort_keys( k,key,m,n)
{
    clear_array(sorted_keys)

    n = 0
    for (key in key_value_pair)
    {
	n++
	sorted_keys[n] = key
    }
    for (k = 1; k < n; ++k)
    {
	for (m = k + 1; m <= n; ++m)
	{
	    if (tolower(sorted_keys[k]) > tolower(sorted_keys[m]))
	    {
		key = sorted_keys[m]
		sorted_keys[m] = sorted_keys[k]
		sorted_keys[k] = key
	    }
	}
    }
}

function squeeze( kbrace,kspace)
{
    sub(/^[ \t]*@[ \t]*/,"@")	# eliminate space before and after initial @
    kbrace = index($0,"{")	# eliminate space between entryname and brace
    kspace = match($0,"[ \t]")
    if (kspace < kbrace)	# then found intervening space
	sub(/[ \t]+{/,"{")	# NB: sub(), NOT gsub(), here
}

function trim()
{
    sub(/[ \t]+$/,"")
}

function warning(msg)
{
    message("%%" msg)
}
'
# Use GNU gawk instead of nawk: Sun Solaris 2.x nawk often complains
# `input record too long'.
gawk \
	-v ALPHABETICAL=$ALPHABETICAL \
	-v BIMONTHLY=$BIMONTHLY \
	-v CHECKMISSING=$CHECKMISSING \
	-v MONTHLY=$MONTHLY \
	-v QUARTERLY=$QUARTERLY \
	-v MONTHNAMES="$MONTHNAMES" \
	"$PROGRAM" \
	$FILES
################################[The End]###############################
